package ru.yandex.direct.mysql.ytsync.synchronizator.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;

import javax.annotation.Nullable;

import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.mysql.MySQLColumnData;
import ru.yandex.direct.mysql.MySQLSimpleRow;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class YtSyncUtil {
    private static final String LINK_TARGET_PATH = "target_path";

    public static <T> CompletableFuture<List<T>> collectAllOrdered(
            List<CompletableFuture<List<T>>> futures) {
        return YtSyncCommonUtil.allFutures(futures).thenApply(unused -> {
            List<T> allResults = new ArrayList<>();
            for (CompletableFuture<List<T>> future : futures) {
                allResults.addAll(future.getNow(null));
            }
            return allResults;
        });
    }

    @FunctionalInterface
    public interface InterruptibleRunnable {
        void run() throws InterruptedException;
    }

    /**
     * Перезапускает r при возникновении InterruptedException, восстанавливает interrupt флаг
     */
    public static void doUninterruptibly(InterruptibleRunnable r) {
        boolean interrupted = false;
        while (true) {
            try {
                r.run();
                break;
            } catch (InterruptedException ie) {
                interrupted = true;
            }
        }
        if (interrupted) {
            Thread.currentThread().interrupt();
        }
    }

    /**
     * Форматирует миллисекунды в виде строки "..d..h..m..s..ms"
     */
    public static String formatMillis(long millis) {
        boolean negative = false;
        if (millis < 0) {
            negative = true;
            millis = -millis;
        }
        List<String> parts = new ArrayList<>(6);
        parts.add(String.format(millis >= 1000L ? "%03dms" : "%dms", millis % 1000L));
        long seconds = millis / 1000L;
        if (seconds > 0) {
            parts.add(String.format(seconds >= 60L ? "%02ds" : "%ds", seconds % 60L));
        }
        long minutes = seconds / 60L;
        if (minutes > 0) {
            parts.add(String.format(minutes >= 60L ? "%02dm" : "%dm", minutes % 60L));
        }
        long hours = minutes / 60L;
        if (hours > 0) {
            parts.add(String.format(hours >= 24L ? "%02dh" : "%dh", hours % 24L));
        }
        long days = hours / 24L;
        if (days > 0) {
            parts.add(String.format("%dd", days));
        }
        if (negative) {
            parts.add("-");
        }
        Collections.reverse(parts);
        return String.join("", parts);
    }

    /**
     * Работает похожим на future.join образом, но в случае interrupt'а потока отменяет future
     */
    public static <T> T joinInterruptibly(CompletableFuture<T> future) throws InterruptedException {
        try {
            return future.get();
        } catch (InterruptedException e) {
            future.cancel(false);
            throw e;
        } catch (ExecutionException e) {
            Throwable x = e.getCause();
            if (x instanceof CompletionException) {
                throw (CompletionException) x;
            } else {
                throw new CompletionException(x);
            }
        }
    }


    /**
     * Получить путь на который указывает ссылка
     *
     * @see <a href=https://wiki.yandex-team.ru/yt/userdoc/softlinks/#detalirealizacii>Символические ссылки</a>
     */
    @Nullable
    public static String readLinkTargetPath(YtSupport yt, String linkPath) {
        //амперсанд нужен для отключения переадресации, чтобы можно было прочитать атрибут ссылки, а не целевого объекта
        String attrPath = linkPath + "&/@" + LINK_TARGET_PATH;
        try (TraceProfile ignored = Trace.current().profile("yt:getLinkTargetPath")) {
            if (!yt.exists(attrPath).join()) { // IGNORE-BAD-JOIN DIRECT-149116
                return null;
            }
            return yt.getNode(attrPath).join().stringValue(); // IGNORE-BAD-JOIN DIRECT-149116
        }
    }

    public static String getSourceColumnValue(EnvironmentType environmentType, String dbName) {
        return environmentType.name().toLowerCase() + ":" + dbName;
    }

    public static boolean isKeywordOfBidsBase(MySQLSimpleRow row) {
        Optional<MySQLColumnData> bidTypeColumnValue = row.stream()
                .filter(columnValue -> columnValue.getSchema().getName().equals("bid_type"))
                .findFirst();
        //1 — keyword
        return bidTypeColumnValue.isPresent() && bidTypeColumnValue.get().getRawValue().equals(1);
    }
}
