package ru.yandex.direct.libs.oauth.client;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.JsonParsableRequest;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.libs.oauth.client.model.ApplicationInfo;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static java.util.Collections.emptyMap;

/**
 * Простой клиент к нескольким методам oauth.yandex.ru
 */
@ParametersAreNonnullByDefault
public class OauthClient {
    public static final String DEFAULT_API_URL = "https://oauth.yandex.ru/";

    private final ParallelFetcherFactory parallelFetcherFactory;
    private final String apiUrl;

    /**
     * Создать клиента
     *
     * @param apiUrl          - базовый урл (должен заканчиваться на /)
     * @param asyncHttpClient - http-client
     * @param fetcherSettings - настройки запросов
     */
    public OauthClient(String apiUrl, AsyncHttpClient asyncHttpClient, FetcherSettings fetcherSettings) {
        this.parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, fetcherSettings);
        this.apiUrl = apiUrl;
    }

    /**
     * Создать клиента со стандартным базовым урлом и политикой запросов
     */
    public static OauthClient createDefault(AsyncHttpClient asyncHttpClient) {
        return new OauthClient(
                DEFAULT_API_URL,
                asyncHttpClient,
                new FetcherSettings()
                        .withParallel(5)
                        .withConnectTimeout(Duration.ofMillis(300))
                        .withSoftTimeout(Duration.ofMillis(600))
                        .withRequestRetries(3)
                        .withTotalRetriesCoef(1)
        );
    }

    /**
     * Получить информацию OAuth-приложениях по id
     */
    public Map<String, Result<ApplicationInfo>> getApplicationsInfo(Collection<String> applicationIds) {
        return doRequest(
                new ArrayList<>(new HashSet<>(applicationIds)),
                applicationId -> apiUrl + "client/" + applicationId + "/info?format=json&locale=ru",
                ApplicationInfo.class,
                "get_apps_info");
    }

    private <T, R> Map<T, Result<R>> doRequest(List<T> params, Function<T, String> urlBuilder, Class<R> responseClass, String profileFunc) {
        if (params.isEmpty()) {
            return emptyMap();
        }

        Map<Long, T> idx2param = EntryStream.of(params)
                .mapKeys(Long::valueOf)
                .toMap();

        List<JsonParsableRequest<R>> reqs = EntryStream.of(idx2param)
                .mapKeyValue((idx, param) -> createRequest(idx, urlBuilder.apply(param), responseClass))
                .toList();

        try (
                TraceProfile profile = Trace.current().profile("oauth:" + profileFunc, "", reqs.size());
                ParallelFetcher<R> parallelFetcher = parallelFetcherFactory.getParallelFetcher()
        ) {
            Map<Long, Result<R>> result = parallelFetcher.execute(reqs);

            return EntryStream.of(result)
                    .mapKeys(idx2param::get)
                    .toMap();
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    private <T> JsonParsableRequest<T> createRequest(long idx, String url, Class<T> cls) {
        return new JsonParsableRequest<>(idx,
                new RequestBuilder("GET")
                        .setUrl(url)
                        .build(),
                cls
        );
    }

}
