package ru.yandex.direct.operation;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Идентификатор добавленного объекта. Используется в тех случаях когда необходимо знать
 * был ли реально добавлен, изменен или является дубликатом объект с данным идентификатором.
 * Актуально для Vcard-ов и возможно для SitelinkSet-ов
 */
@ParametersAreNonnullByDefault
public final class AddedModelId {

    private enum Status {
        NEW("new"),
        CHANGED("changed"),
        EXISTING("existing"),
        /**
         * Используется в случае, когда нам неизвестно был ли добавлен, изменен или является дубликатом объект c
         * данным идентификатором.
         */
        UNKNOWN("unknown");

        private final String name;

        Status(String name) {
            this.name = name;
        }

        @Override
        public String toString() {
            return name;
        }
    }

    /**
     * Идентификатор объекта
     */
    private final Long id;

    private final Status status;

    private AddedModelId(Long id, Status status) {
        this.id = Objects.requireNonNull(id, "id");
        this.status = status;
    }

    public static AddedModelId ofNew(Long id) {
        return new AddedModelId(id, Status.NEW);
    }

    public static AddedModelId ofExisting(Long id) {
        return new AddedModelId(id, Status.EXISTING);
    }

    public static AddedModelId ofChanged(Long id) {
        return new AddedModelId(id, Status.CHANGED);
    }

    public static AddedModelId ofUnknown(Long id) {
        return new AddedModelId(id, Status.UNKNOWN);
    }

    @Override
    public String toString() {
        return "AddedModelId{" +
                "id=" + id +
                ", status=" + status +
                '}';
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        AddedModelId that = (AddedModelId) o;

        return id.equals(that.id);
    }

    public Long getId() {
        return id;
    }

    public boolean isAdded() {
        return status == Status.NEW;
    }

    public boolean isChanged() {
        return status == Status.CHANGED;
    }

    public boolean isExisting() {
        return status == Status.EXISTING;
    }
}
