package ru.yandex.direct.operation;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;

import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.FunctionalUtils.batchApply;
import static ru.yandex.direct.utils.FunctionalUtils.batchDispatch;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResultUtils.hasAnyErrors;
import static ru.yandex.direct.validation.result.ValidationResultUtils.mergeValidationResults;

@ParametersAreNonnullByDefault
public class FunctionOperation<I, O> implements PartiallyApplicableOperation<O> {
    private final Function<List<I>, List<ValidationResult<I, Defect>>> validator;
    private final Function<List<I>, List<O>> function;
    private final List<I> inputList;
    private final Applicability applicability;

    private MassResult<O> result;
    private List<? extends ValidationResult<? extends I, Defect>> validationResultList;

    public FunctionOperation(Function<List<I>, List<O>> function, List<I> inputList) {
        this(Applicability.PARTIAL, items -> mapList(items, ValidationResult::success), function, inputList);
    }

    public FunctionOperation(
            Applicability applicability,
            Function<List<I>, List<ValidationResult<I, Defect>>> validator,
            Function<List<I>, List<O>> function,
            List<I> inputList) {
        this.applicability = applicability;
        this.validator = validator;
        this.function = function;
        this.inputList = inputList;
        this.result = null;
    }

    @Override
    public Optional<MassResult<O>> prepare() {
        validationResultList = batchApply(validator, inputList);

        if (isFull(applicability) && hasAnyErrors(validationResultList)) {
            result = MassResult.brokenMassAction(
                    Collections.nCopies(inputList.size(), null),
                    mergeValidationResults(inputList, validationResultList));
            return Optional.of(result);
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Set<Integer> getValidElementIndexes() {
        return EntryStream.of(validationResultList)
                .filterValues(not(ValidationResult::hasAnyErrors))
                .keys()
                .toSet();
    }

    /**
     * Применяет изменения ко всем валидным элементам операции
     */
    @Override
    public MassResult<O> apply() {
        checkState(result == null, "operation is already applied");
        return apply(getValidElementIndexes());
    }

    /**
     * Применяет изменения к валидным элементам, если их индекс содержится elementIndexesToApply. Позволяет
     * применить операцию только к некоторому подмножеству элементов
     *
     * @return для невалидных элементов возвращает BROKEN результат с результатом валидации, для валидных и
     * применённых -- SUCCESSFUL с результатом, для остальных -- CANCELED
     */
    @Override
    public MassResult<O> apply(Set<Integer> elementIndexesToApply) {
        checkState(result == null, "operation is already applied");
        checkArgument(Sets.difference(elementIndexesToApply, getValidElementIndexes()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");

        var validatedItems = createValidatedItems(inputList, validationResultList);

        List<Result<O>> results = batchDispatch(
                validatedItems, List.copyOf(elementIndexesToApply),
                this::applyAndCreateResults, this::createCanceledOrBrokenResults);
        result = new MassResult<>(results, ValidationResult.success(inputList), ResultState.SUCCESSFUL);
        return result;
    }

    /**
     * Отменяет операцию без применения.
     * @return для всех невалидных элементов вернёт BROKEN результат, для остальных CANCELED
     */
    @Override
    public MassResult<O> cancel() {
        return apply(Collections.emptySet());
    }

    @Override
    public Optional<MassResult<O>> getResult() {
        return Optional.ofNullable(result);
    }

    private List<Result<O>> applyAndCreateResults(List<ValidatedItem<I>> validatedItem) {
        List<O> result = function.apply(mapList(validatedItem, ValidatedItem::getItem));
        checkState(validatedItem.size() == result.size(), "function must returns result for each input object");
        return EntryStream.zip(validatedItem, result)
                .mapKeyValue(this::createSuccessfulResult)
                .toList();
    }

    private List<Result<O>> createCanceledOrBrokenResults(List<ValidatedItem<I>> validatedItem) {
        return batchDispatch(
                validatedItem, ValidatedItem::hasAnyErrors,
                this::createBrokenResults, this::createCanceledResults);
    }

    private List<Result<O>> createBrokenResults(List<ValidatedItem<I>> validatedItems) {
        return mapList(validatedItems, this::createBrokenResult);
    }

    private List<Result<O>> createCanceledResults(List<ValidatedItem<I>> validatedItems) {
        return mapList(validatedItems, this::createCanceledResult);
    }


    private Result<O> createSuccessfulResult(ValidatedItem<?> validatedItem, O resultForItem) {
        return Result.successful(resultForItem, validatedItem.getValidationResult());
    }

    private Result<O> createCanceledResult(ValidatedItem<?> validatedItem) {
        return Result.canceled(validatedItem.getValidationResult());
    }

    private Result<O> createBrokenResult(ValidatedItem<?> validatedItem) {
        return Result.broken(validatedItem.getValidationResult());
    }

    private List<ValidatedItem<I>> createValidatedItems(List<I> inputList,
                                                        List<? extends ValidationResult<? extends I, Defect>> validationResultList) {
        return EntryStream.zip(inputList, validationResultList)
                .mapKeyValue(ValidatedItem::new)
                .toList();
    }

    private static class ValidatedItem<I> {

        private final I item;
        private final ValidationResult<?, Defect> validationResult;

        private ValidatedItem(I item, ValidationResult<?, Defect> validationResult) {
            this.item = item;
            this.validationResult = validationResult;
        }

        public I getItem() {
            return item;
        }

        public ValidationResult<?, Defect> getValidationResult() {
            return validationResult;
        }

        public boolean hasAnyErrors() {
            return validationResult.hasAnyErrors();
        }
    }
}
