package ru.yandex.direct.operation;

import java.util.Optional;

import ru.yandex.direct.result.MassResult;

/**
 * Интерфейс пошаговой операции. Операция состоит из двух шагов:
 * подготовка ({@link #prepare()}) и применение изменений ({@link #apply()}).
 * <p>
 * На этапе подготовки проводится валидация, а также могут выполняться
 * любые дополнительные действия, необходимые как для валидации,
 * так и для последующего применения изменений, но действия эти
 * не должны быть модифицирующими. Пример таких действий: предобработка
 * входных данных, вычисление дополнительных изменений в смежных объектах.
 * <p>
 * Если валидация прошла успешно и изменения могут быть применены,
 * то метод {@link #prepare()} возвращает пустой объект {@link Optional}.
 * В противном случае метод возвращает объект {@link Optional} с результатом
 * операции {@link MassResult}, в котором содержится результат валидации
 * с ошибками, из-за которых изменения не могут быть применены.
 * <p>
 * Если валидация прошла успешно (метод {@link #prepare()} вернул
 * пустой объект {@link Optional}), то изменения можно применить
 * с помощью метода {@link #apply()}, который возвращает результат
 * операции.
 * <p>
 * Если пошаговое выполнение операции не требуется, то
 * можно вызвать метод {@link #prepareAndApply()},
 * который вернет результат операции (положительный или отрицательный
 * в зависимости от результата валидации).
 *
 * @param <R> тип параметра возвращаемого объекта {@link MassResult}
 */
public interface Operation<R> {

    Optional<MassResult<R>> prepare();

    MassResult<R> apply();

    default MassResult<R> prepareAndApply() {
        return prepare().orElseGet(this::apply);
    }

    /**
     * Отменить операцию, не выполняя изменений в базе. Операция не может быть продолжена после отмены
     * <p>
     * Если операция уже вернула результат при вызове {@link #prepare()}, cancel() вызываться не должен.
     * Отменить операцию можно отменить только после вызова {@link #prepare()}
     *
     * @return результат операции, в котором все элементы {@link MassResult} содержат null, а элементы без ошибок должны
     * быть помечены как отменённые.
     */
    MassResult<R> cancel();

    Optional<MassResult<R>> getResult();
}
