package ru.yandex.direct.operation.add;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ContiguousSet;
import com.google.common.collect.DiscreteDomain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Range;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.PartiallyApplicableModelOperation;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItemsWithIndex;
import static ru.yandex.direct.validation.util.ValidationUtils.cloneValidationResultSubNodesWithIssues;

/**
 * Пошаговая операция создания (добавления) списка объектов.
 * <p>
 * Смотреть документацию к {@link Operation}.
 * <p>
 * Позволяет выполнить добавление в два связанных между собой этапа:
 * подготовка (валидация + доп. действия) и создание провалидированных объектов в БД.
 * <p>
 * Свойство {@link #applicability} управляет возможностью выполнения операции над валидными
 * объектами при наличии невалидных. Если свойство равно {@code PARTIAL}, то при наличии
 * в исходном списке валидных и невалидных объектов, этап подготовки считается пройденным
 * успешно, т.е. метод {@link #prepare()} вернет пустой {@link Optional}, и последующий
 * вызов метода {@link #apply()} выполнит действие над валидными моделями. Но если все
 * модели будут невалидны, то метод {@link #prepare()} вернет результат и выполнить
 * операцию будет невозможно. Если свойство равно {@code FULL}, то операция может быть
 * выполнена только когда все объекты в списке валидны, то есть при наличии хотя бы
 * одного невалидного объекта метод {@link #prepare()} вернет результат.
 *
 * @param <M> тип модели
 */
public abstract class AbstractAddOperation<M extends ModelWithId, R> implements PartiallyApplicableModelOperation<M, R> {

    protected final Applicability applicability;
    private final List<M> models;
    protected ValidationResult<List<M>, Defect> validationResult;
    private Map<Integer, M> preValidModelsMap;
    private Map<Integer, M> validModelsMap;

    private boolean prepared;
    private boolean executed;
    private MassResult<R> result;

    public AbstractAddOperation(Applicability applicability, List<M> models) {
        this.applicability = applicability;
        this.models = ImmutableList.copyOf(models);
    }

    public List<M> getModels() {
        return models;
    }

    public ValidationResult<List<M>, Defect> getValidationResult() {
        return validationResult;
    }

    @Override
    public Optional<MassResult<R>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        preValidateInternal();
        if (result != null) {
            return Optional.of(result);
        }
        onPreValidated(new ModelsPreValidatedStepInner());

        validateInternal();
        if (result != null) {
            return Optional.of(result);
        }
        onModelsValidated(new ModelsValidatedStepInner());

        return Optional.empty();
    }

    @Override
    public MassResult<R> apply() {
        checkApplyOrCancelPreconditions();
        Set<Integer> allIndexesSet = ContiguousSet.create(
                Range.closedOpen(0, models.size()), DiscreteDomain.integers());
        return applyInternal(allIndexesSet);
    }

    @Override
    public MassResult<R> apply(Set<Integer> elementIndexesToApply) {
        checkApplyOrCancelPreconditions();
        checkArgument(Sets.difference(elementIndexesToApply, validModelsMap.keySet()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");
        return applyInternal(elementIndexesToApply);
    }

    private MassResult<R> applyInternal(Set<Integer> elementIndexesToApply) {
        executed = true;

        Map<Integer, M> validModelsMapToApply = EntryStream.of(validModelsMap)
                .filterKeys(elementIndexesToApply::contains)
                .toMap();

        beforeExecution(validModelsMapToApply);

        result = executeInternal(validModelsMapToApply);

        onExecuted(validModelsMapToApply);

        return result;
    }

    @Override
    public final MassResult<R> cancel() {
        checkApplyOrCancelPreconditions();
        cancelInternal();
        return result;
    }

    @Override
    public Optional<MassResult<R>> getResult() {
        return Optional.ofNullable(result);
    }

    @Override
    public Set<Integer> getValidElementIndexes() {
        checkState(prepared, "prepare() must be called before getValidElementIndexes()");

        if (validModelsMap != null) {
            return validModelsMap.keySet();
        } else {
            return emptySet();
        }
    }

    protected boolean isPrepared() {
        return prepared;
    }

    protected boolean isPrepareSuccessful() {
        return prepared && (result == null || result.isSuccessful()) || executed;
    }

    protected boolean isExecuted() {
        return executed;
    }

    protected <T, X extends ModelWithId> void modifyProperty(ModelProperty<? super X, T> modelProperty,
                                                             Map<Integer, T> indexValueMap) {
        indexValueMap.keySet().forEach(i -> assertModificationIsAcceptable(i, modelProperty));
        for (Map.Entry<Integer, T> entry : indexValueMap.entrySet()) {
            int i = entry.getKey();
            M model = models.get(i);
            T value = entry.getValue();
            @SuppressWarnings("unchecked")
            X castedModel = (X) model;
            modelProperty.set(castedModel, value);
        }
    }

    private void cancelInternal() {
        executed = true;
        List<R> results = Collections.nCopies(models.size(), null);
        // все валидные элементы помечаем отменёнными, т.к. для них не выполнялась операция
        result = MassResult.successfulMassAction(results, validationResult, validModelsMap.keySet());
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
    }

    private void assertModificationIsAcceptable(int changesIndex, ModelProperty<? extends Model, ?> property) {
        M model = models.get(changesIndex);
        checkArgument(isPropertyModificationAcceptable(property, model),
                "No such property in the model at specified index.%n"
                        + "index: %d, model type: %s, property name: %s",
                changesIndex, model.getClass().getSimpleName(), property.name());
    }

    private boolean isPropertyModificationAcceptable(ModelProperty<? extends Model, ?> property, M model) {
        return property.getModelClass().isAssignableFrom(model.getClass());
    }

    private boolean isModificationAcceptable(ModelProperty<? extends Model, ?> property, M model) {
        return property.getModelClass().isAssignableFrom(model.getClass());
    }

    private void preValidateInternal() {
        validationResult = preValidate(models);
        validationResult = cloneValidationResultSubNodesWithIssues(validationResult);
        Map<Integer, M> preValidModelsMap = getValidItemsWithIndex(validationResult);

        if (validationResult.hasErrors() ||
                preValidModelsMap.isEmpty() ||
                (preValidModelsMap.size() < models.size() && isFull(applicability))) {
            this.result = MassResult.brokenMassAction(mapList(models, m -> null), validationResult);
            return;
        }

        this.preValidModelsMap = preValidModelsMap;
    }

    /**
     * метод предварительной валидации списка объектов. не обязателен к реализации
     *
     * @param models исходный список объектов для создания в БД.
     * @return результат валидации списка объектов
     */
    protected ValidationResult<List<M>, Defect> preValidate(List<M> models) {
        return new ValidationResult<>(models);
    }

    /**
     * Коллбэк, вызываемый, после предварительной валидации списка объектов, если результат валидации
     * позволяет выполнить операцию. Может быть реализован в потомке при необходимости
     * выполнить дополнительные действия в описанной ситуации, например, вычислить
     * необходимые дополнительные действия, которые необходимо совершить после применения
     * изменений. Метод не должен модифицировать состояние БД, так как является частью подготовки.
     * <p>
     * Через входной параметр доступны: результат предварительной валидации, список валидных объектов.
     *
     * @param modelsPreValidatedStep объект для доступа к результату предварительной валидации и списку валидных
     *                               объектов.
     */
    @SuppressWarnings("unused")
    protected void onPreValidated(ModelsPreValidatedStep<M> modelsPreValidatedStep) {
    }

    private void validateInternal() {
        validate(validationResult);

        if (validationResult.hasErrors()) {
            this.result = MassResult.brokenMassAction(mapList(models, m -> null), validationResult);
            return;
        }

        List<M> validModels = getValidItems(validationResult);

        if (validModels.isEmpty() ||
                (validModels.size() < models.size() && isFull(applicability))) {
            this.result = MassResult.brokenMassAction(mapList(models, m -> null), validationResult);
            return;
        }

        this.validModelsMap = getValidItemsWithIndex(validationResult);
    }

    /**
     * Абстрактный метод валидации списка объектов, должен быть реализован в потомке.
     *
     * @param preValidationResult результат предварительной валидации.
     */
    protected abstract void validate(ValidationResult<List<M>, Defect> preValidationResult);

    /**
     * Коллбэк, вызываемый, после валидации списка объектов, если результат валидации
     * позволяет выполнить операцию. Может быть реализован в потомке при необходимости
     * выполнить дополнительные действия в описанной ситуации, например, вычислить
     * необходимые дополнительные действия, которые необходимо совершить после применения
     * изменений. Метод не должен модифицировать состояние БД, так как является частью подготовки.
     * <p>
     * Через входной параметр доступны: результат валидации, список валидных объектов.
     *
     * @param modelsValidatedStep объект для доступа к результату валидации и списку валидных объектов.
     */
    @SuppressWarnings("unused")
    protected void onModelsValidated(ModelsValidatedStep<M> modelsValidatedStep) {
    }

    private MassResult<R> executeInternal(Map<Integer, M> validModelsMapToApply) {
        Map<Integer, R> addedResults = execute(validModelsMapToApply);

        checkState(addedResults.keySet().equals(validModelsMapToApply.keySet()), "execute() returned map with "
                + "invalid keySet (should be equals)");

        Set<Integer> canceledElements = Sets.difference(validModelsMap.keySet(), validModelsMapToApply.keySet());
        return createMassResult(addedResults, validationResult, canceledElements);
    }

    /**
     * Выполняется непосредственно перед исполнением операции.
     * Метод можно реализовать в потомке для выполнения дополнительных действий с {@code validModelsMapToApply}.
     * Список моделей, передающийся в этот метод точно такой же с которым выполняется {@link #execute(Map)}.
     *
     * @param validModelsMapToApply маппинг индексов в исходном массиве на модели, к которым должна примениться опреация
     */
    protected void beforeExecution(Map<Integer, M> validModelsMapToApply) {
    }

    /**
     * Абстрактный метод создания указанных валидных объектов в базе данных.
     * Входной {@code validModelsMapToApply} может содержать не все валидные модели, а только те,
     * к которым должна примениться операция (в случае частичного исполнения согласно
     * {@link PartiallyApplicableOperation}).
     * Размер входного map должен быть равен выходному и соответствовать ему по ключам
     *
     * @param validModelsMapToApply маппинг индексов в исходном массиве на модели, к которым должна примениться опреация
     * @return Map, соответствующий по размеру и ключам validModelsMap, содержащий объект ре>зультата
     * для добавленных объектов
     */
    protected abstract Map<Integer, R> execute(Map<Integer, M> validModelsMapToApply);

    /**
     * Создать результат выполнения операции.
     * <p>
     * Предполагается, что на данный момент операция выполнена
     *
     * @param resultMap              Map добавленных объектов
     * @param validationResult       Объединенный результат валидации объектов
     * @param canceledElementIndexes Индексы объектов для которых операция отменилась
     */
    protected MassResult<R> createMassResult(Map<Integer, R> resultMap,
                                             ValidationResult<List<M>, Defect> validationResult,
                                             Set<Integer> canceledElementIndexes) {
        List<R> results = new ArrayList<>();
        for (int i = 0; i < models.size(); i++) {
            results.add(resultMap.get(i));
        }
        return MassResult.successfulMassAction(results, validationResult, canceledElementIndexes);
    }

    /**
     * Коллбэк, вызываемый после применения изменений (вызова метода {@link #execute(Map)}.
     * Может быть реализован в потомке для выполнения дополнительных действий, например,
     * модификации связанных объектов. Вычисление этих дополнительных действий должно
     * производиться заранее, в методе {@link #onModelsValidated(ModelsValidatedStep)},
     * а здесь - только непосредственное выполнение.
     *
     * @param validModelsMapToApply маппинг индексов в исходном массиве на модели, к которым применилась опреация
     */
    protected void onExecuted(Map<Integer, M> validModelsMapToApply) {
    }

    private class ModelsPreValidatedStepInner implements ModelsPreValidatedStep<M> {
        @Override
        public List<M> getModels() {
            return models;
        }

        @Override
        public ValidationResult<List<M>, Defect> getValidationResult() {
            return validationResult;
        }

        @Override
        public Map<Integer, M> getPreValidModelsMap() {
            return preValidModelsMap;
        }
    }

    private class ModelsValidatedStepInner extends ModelsPreValidatedStepInner implements ModelsValidatedStep<M> {
        @Override
        public Map<Integer, M> getValidModelsMap() {
            return validModelsMap;
        }
    }
}

