package ru.yandex.direct.operation.aggregator;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.operation.execution.PartiallyInParallelExecutionStrategy;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonMap;
import static org.apache.commons.collections4.CollectionUtils.emptyCollection;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

/**
 * Агрегатор операций {@link PartiallyApplicableOperation}
 * <p>
 * Этот агрегатор контролирует процесс выполнения нескольких операций. Агрегатор инициализируется операциями,
 * которые нужно выполнить совместно. Есть основная операция, из результата которой будет браться идентификаторы,
 * и остальные операции.
 * <p>
 * Каждая операция выполняется над списком элементов. Элементы с одним индексом в разных
 * операциях представляют собой общую логическую сущность, и результаты их валидации должны рассматриваться совместно.
 * <p>
 * Операции соответствует ключ, который будет использоваться для объединения результатов операций. Т.е. в каждом
 * элементе результата совместной операции, пути ошибок (если они имели место) будут
 * начинаться с ключа операции, где эта ошибка обнаружилась.
 * <p>
 * Например, есть две операции Op1 и Op2, c ключами соответственно k1 и k2. Если при валидации в Op1 возникнет ошибка
 * для какого либо элемента с путем p1, то в совместном результате эта ошибка будет с путём k1 + p1.
 * <p>
 * Внимание! ValidationResult, находящийся в возвращаемом значении, содержит в качестве значений идентификаторы
 * из основной операции, а не исходные объекты
 */
@ParametersAreNonnullByDefault
public class PartiallyApplicableOperationAggregator<R> {
    private final Map<String, PartiallyApplicableOperation<R>> operationMap;
    private final String masterKey;

    private PartiallyApplicableOperationAggregator(String masterKey,
                                                   Map<String, PartiallyApplicableOperation<R>> operationMap) {
        checkArgument(operationMap.containsKey(masterKey));
        this.masterKey = masterKey;
        this.operationMap = operationMap;
    }

    /**
     * Не проверяет факт применения операций над списками одинаковых размеров! Будьте бдительны!
     * Проверяйте длины входящих данных операций самостоятельно!
     *
     * @deprecated
     */
    @Deprecated
    public static <R> PartiallyApplicableOperationAggregator<R> of(
            String masterKey, PartiallyApplicableOperation<R> masterOperation,
            String operationKey2, PartiallyApplicableOperation<R> operation2) {
        return new PartiallyApplicableOperationAggregator<>(
                masterKey,
                ImmutableMap.of(masterKey, masterOperation, operationKey2, operation2));
    }

    /**
     * Не проверяет факт применения операций над списками одинаковых размеров! Будьте бдительны!
     * Проверяйте длины входящих данных операций самостоятельно!
     *
     * @deprecated
     */
    @Deprecated
    public static <R> PartiallyApplicableOperationAggregator<R> of(
            String masterKey, PartiallyApplicableOperation<R> masterOperation,
            String operationKey2, PartiallyApplicableOperation<R> operation2,
            String operationKey3, PartiallyApplicableOperation<R> operation3) {
        return new PartiallyApplicableOperationAggregator<>(
                masterKey,
                ImmutableMap.of(masterKey, masterOperation, operationKey2, operation2, operationKey3, operation3));
    }

    public static <R, I1, I2> PartiallyApplicableOperationAggregator<R> of(
            String masterKey, List<I1> items1,
            OperationCreator<I1, PartiallyApplicableOperation<R>> masterOperationCreator,
            String operationKey2, List<I2> items2,
            OperationCreator<I2, PartiallyApplicableOperation<R>> operationCreator2) {
        checkArgument(items1.size() == items2.size());
        return new PartiallyApplicableOperationAggregator<>(
                masterKey,
                Map.of(
                        masterKey, masterOperationCreator.create(items1),
                        operationKey2, operationCreator2.create(items2)
                ));
    }

    public static <R, I1, I2, I3> PartiallyApplicableOperationAggregator<R> of(
            String masterKey, List<I1> items1,
            OperationCreator<I1, PartiallyApplicableOperation<R>> masterOperationCreator,
            String operationKey2, List<I2> items2,
            OperationCreator<I2, PartiallyApplicableOperation<R>> operationCreator2,
            String operationKey3, List<I3> items3,
            OperationCreator<I3, PartiallyApplicableOperation<R>> operationCreator3) {
        checkArgument(items1.size() == items2.size() && items2.size() == items3.size());
        return new PartiallyApplicableOperationAggregator<>(
                masterKey,
                Map.of(
                        masterKey, masterOperationCreator.create(items1),
                        operationKey2, operationCreator2.create(items2),
                        operationKey3, operationCreator3.create(items3)
                ));
    }

    /**
     * При вызове метода выполняется последовательно:
     * <ul>
     * <li>подготовка каждой операции</li>
     * <li>определение элементов, которые валидны в каждой операции</li>
     * <li>выполнение операций, только для валидных везде элементов</li>
     * <li>объединение результатов всех операций</li>
     * </ul>
     *
     * @return Объединённый результат всех операций
     */
    public MassResult<R> prepareAndApplyPartialTogether() {
        new PartiallyInParallelExecutionStrategy().execute(operationMap.values());
        Map<String, PartiallyApplicableOperation<R>> otherOperationMap = EntryStream.of(operationMap)
                .removeKeys(masterKey::equals)
                .toMap();
        return mergeMassResults(null, masterKey, getMasterOperation(), ImmutableMap.copyOf(otherOperationMap));
    }

    private PartiallyApplicableOperation<R> getMasterOperation() {
        return operationMap.get(masterKey);
    }

    public static <I, R> MassResult<R> mergeMassResults(
            @Nullable List<I> operationInput,
            String masterKey, PartiallyApplicableOperation<R> masterOperation,
            Map<String, PartiallyApplicableOperation<?>> otherOperations) {
        List<R> items = masterOperation.getResult()
                .map(Result::getResult)
                .map(results -> mapList(results, Result::getResult))
                .orElse(emptyList());

        LinkedHashMap<String, MassResult<?>> massResultMap = EntryStream.of(otherOperations)
                .prepend(singletonMap(masterKey, masterOperation))
                .mapValues(Operation::getResult)
                .<MassResult<?>>mapValues(Optional::get)
                .toCustomMap(LinkedHashMap::new);

        ResultState massResultState = getAggregatedMassResultState(massResultMap.values());
        if (massResultState == ResultState.BROKEN) {
            return createBrokenMassResult(massResultMap);
        } else {
            return createSuccessfulMassResult(operationInput, items, massResultMap);
        }
    }

    @Nonnull
    private static LinkedHashMap<String, Result<?>> getResultsForIndex(
            LinkedHashMap<String, MassResult<?>> massResultMap, int i) {
        return EntryStream.of(massResultMap).<Result<?>>mapValues(m -> m.get(i)).toCustomMap(LinkedHashMap::new);
    }

    @Nonnull
    private static <R> MassResult<R> createBrokenMassResult(LinkedHashMap<String, MassResult<?>> massResultMap) {
        // В случае ошибки операции, проще вернуть результат валидации только из одной операции.
        Optional<Map.Entry<String, MassResult<?>>> maybeBrokenVr = EntryStream.of(massResultMap)
                .filterValues(r -> r.getState() == ResultState.BROKEN)
                .findFirst();
        checkState(maybeBrokenVr.isPresent());
        MassResult<?> brokenMassResult = maybeBrokenVr.get().getValue();
        ValidationResult<?, Defect> brokenVr = brokenMassResult.getValidationResult();
        checkState(brokenVr != null && brokenVr.getValue() instanceof List,
                "MassResult must contains ValidationResult of List");
        @SuppressWarnings("unchecked")
        List<Object> values = (List<Object>) brokenVr.getValue();
        ValidationResult<List<Object>, Defect> resultBrokenVr =
                new ValidationResult<>(values, brokenVr.getErrors(),
                        brokenVr.getWarnings());
        return MassResult.brokenMassAction(null, resultBrokenVr);
    }

    @Nonnull
    private static <I, R> MassResult<R> createSuccessfulMassResult(@Nullable List<I> operationInput,
                                                                   List<R> masterMassResultItems,
                                                                   LinkedHashMap<String, MassResult<?>> operationsMassResultMap) {
        ValidationResult<List<Object>, Defect> mergedVr;
        if (operationInput == null) {
            // входные данные неизвестны (старое поведение)
            //noinspection unchecked
            mergedVr = new ValidationResult<>((List<Object>) masterMassResultItems);
        } else {
            //noinspection unchecked
            mergedVr = new ValidationResult<>((List<Object>) operationInput);
        }
        for (int i = 0; i < masterMassResultItems.size(); i++) {
            ValidationResult<Object, Defect> subVr =
                    mergedVr.getOrCreateSubValidationResult(index(i), mergedVr.getValue().get(i));

            LinkedHashMap<String, Result<?>> results = getResultsForIndex(operationsMassResultMap, i);
            checkAggregatedResultState(results.values());
            Map<String, ? extends ValidationResult<?, Defect>> vrs = EntryStream.of(results)
                    .mapValues(Result::getValidationResult)
                    .filterValues(Objects::nonNull)
                    .toMap();
            // Результаты операций привязываются к результату элемента согласно ключам операций, переданным
            // в конструктор PartiallyApplicableOperationAggregator, для этого создаются фиктивные ноды с
            // путём field(key)
            vrs.forEach((key, vr) -> transferSubNodesWithIssues(
                    vr, subVr.getOrCreateSubValidationResult(field(key), vr.getValue())));
        }
        return MassResult.successfulMassAction(masterMassResultItems, mergedVr);
    }

    /**
     * Определяет объединённое состояние нескольких {@link MassResult}
     *
     * @return общее состояние {@link MassResult}
     */
    private static ResultState getAggregatedMassResultState(Collection<MassResult<?>> allResults) {
        Set<ResultState> resultStates = new HashSet<>(mapList(allResults, MassResult::getState));
        if (resultStates.contains(ResultState.BROKEN)) {
            return ResultState.BROKEN;
        } else {
            return ResultState.SUCCESSFUL;
        }
    }

    /**
     * Проверяёт состояние нескольких {@link Result} для одной записи, полученных из разных операций.
     * <p>
     * Нужно падать в случае если, например, успешно изменили модель Client, но изменения в User не применились.
     * Правильная реализация операций этого не допускает, но операции и результаты приходят извне,
     * поэтому перепроверяем.
     */
    static void checkAggregatedResultState(Collection<Result<?>> allResults) {
        Set<ResultState> resultStates = new HashSet<>(mapList(allResults, Result::getState));
        checkState(!resultStates.contains(ResultState.SUCCESSFUL)
                        || (resultStates.contains(ResultState.SUCCESSFUL) && resultStates.size() == 1)
                        || checkSuccessfulCanBeIgnored(allResults),
                "Invalid combination of Result states");
    }

    /**
     * проверяет ситуацию когда есть ResultState.SUCCESSFUL и какие-то другие ResultState.
     * <p>
     * В таком случае если ResultState.SUCCESSFUL для пустого списка, то его можно игнорировать,
     * поскольку операция была выполнена для пустого списка элементов и изменений в базе нет.
     *
     * @return true если все ResultState.SUCCESSFUL можно игнорировать
     */
    private static boolean checkSuccessfulCanBeIgnored(Collection<Result<?>> allResults) {
        return StreamEx.of(allResults)
                .filter(result -> result.getState() == ResultState.SUCCESSFUL)
                .allMatch(result -> result.getValidationResult().getValue().equals(emptyCollection()));
    }
}
