package ru.yandex.direct.operation.aggregator;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.google.common.collect.MoreCollectors;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.operation.creator.OperationCreators;
import ru.yandex.direct.operation.execution.AllOrNothingExecutionStrategy;
import ru.yandex.direct.operation.execution.ExecutionStrategy;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

/**
 * Агрегатор операций. Принцип его работы в том, что он может разбивать входной набор данных на несколько кусков,
 * которые должны обрабатываться своими операциями, а после выполнения операций собирать общий результат
 * для всего набора данных.
 * <p>
 * Внимание! Мерж сложностью O(N * M) из-за предикатов (N кол-во элементов, М кол-во сплитов (операций). Поэтому не
 * стоит использовать в таком виде для большого кол-ва сплитов.
 * <p>
 * Порядок выполнения операций соответствует порядку вызовов метода
 * {@link SplitAndMergeOperationAggTypedBuilder#addSubOperation}.
 * Это важно, например, при обработке ключевых фраз и автотаргетинга: сначала обрабатываются фразы, а потом
 * автотаргетинг, ставки которого вычисляются в зависимости от добавленных фраз.
 * См. например {@code AddKeywordsDelegate#processList}.
 * <p>
 * Пример создания агрегатора билдером:
 * <pre>
 * {@code
 * SplitAndMergeOperationAggregator<UpdateInputItem, UpdateOutputItem> updateOperation =
 *        SplitAndMergeOperationAggregator.builder()
 *            .addSubOperation(
 *                item -> item instanceof UpdateInputItem.ForKeyword,
 *                updateCoreKeywordOperationCreator)
 *            .addSubOperation(
 *                item -> item instanceof UpdateInputItem.ForRelevanceMatch,
 *                updateCoreRelevanceMatchOperationCreator)
 *            .build();
 *    MassResult<UpdateOutputItem> result = updateOperation.execute(internalRequest);
 * }
 * </pre>
 *
 * @param <I> Тип элемента списка исходных данных операций (Input)
 * @param <O> Тип элемента в результате операций (Output)
 */
public class SplitAndMergeOperationAggregator<I, O> {
    private List<SubOperationDescriptor<I, O>> subOperationDescriptors;
    private ExecutionStrategy<Operation<?>> executionStrategy;

    private SplitAndMergeOperationAggregator(List<SubOperationDescriptor<I, O>> subOperationDescriptors,
                                             boolean forPartialOperations) {
        this.subOperationDescriptors = subOperationDescriptors;
        this.executionStrategy = new AllOrNothingExecutionStrategy(true, forPartialOperations);
    }

    public MassResult<O> execute(List<I> operationInput) {
        return execute(operationInput, false);
    }

    public MassResult<O> validate(List<I> operationInput) {
        return execute(operationInput, true);
    }

    private MassResult<O> execute(List<I> operationInput, boolean validateOnly) {
        List<SubOperationDescriptor<I, O>> descriptorByInputIdx = operationInput.stream()
                .map(this::getSubOperationDescriptorByValue)
                .collect(Collectors.toList());

        Map<SubOperationDescriptor<I, O>, List<I>> operationDescWithInput = EntryStream.of(operationInput)
                .mapKeys(descriptorByInputIdx::get)
                .grouping(IdentityHashMap::new);

        // создаём операции
        Map<SubOperationDescriptor<I, O>, Operation<O>> createdOperations = EntryStream.of(operationDescWithInput)
                .mapToValue(SubOperationDescriptor::createOperation)
                .toCustomMap(IdentityHashMap::new);

        // сортируем операции в порядке добавления применяемых дескрипторов
        List<Operation<O>> operationList = EntryStream.of(createdOperations)
                .sorted(Comparator.comparingInt(e -> e.getKey().ordinal))
                .values()
                .toList();

        // запускаем все операции
        if (validateOnly) {
            executionStrategy.validate(operationList);
        } else {
            executionStrategy.execute(operationList);
        }

        Optional<MassResult<O>> brokenMassResult = StreamEx.of(createdOperations.values())
                .map(Operation::getResult)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .findFirst(not(Result::isSuccessful));

        if (brokenMassResult.isPresent()) {
            // ни одна операция не запускалась только в том случае, если есть ошибка уровня операции
            return brokenMassResult.get();
        }

        // объединяем все результаты всех операций в один результат, который будем возвращать вовне
        checkState(operationList.stream().map(Operation::getResult).allMatch(Optional::isPresent),
                "Все операции должны вернуть результат");

        // достаём результаты операций
        Map<SubOperationDescriptor<I, O>, MassResult<O>> operationResults = EntryStream.of(createdOperations)
                .mapValues(Operation::getResult)
                .mapValues(Optional::get)
                .toCustomMap(IdentityHashMap::new);

        checkState(operationResults.values().stream().allMatch(Result::isSuccessful),
                "Операции должны запускаться только если пройдут валидацию, "
                        + "значит все результаты должны быть успешны");
        int resultNumOverAllOperations =
                operationResults.values().stream().map(Result::getResult).mapToInt(List::size).sum();
        checkState(resultNumOverAllOperations == operationInput.size(),
                "Кол-во результатов от всех операций должно совпадать с кол-вом строк во входных данных");

        Map<SubOperationDescriptor<I, O>, Iterator<Result<O>>> resultIterators = EntryStream.of(operationResults)
                .mapValues(Result::getResult)
                .mapValues(List::iterator)
                .toCustomMap(IdentityHashMap::new);

        List<Result<O>> innerResults = descriptorByInputIdx.stream()
                .map(resultIterators::get)
                .map(Iterator::next)
                .collect(Collectors.toList());
        return new MassResult<>(innerResults, ValidationResult.success(operationInput), ResultState.SUCCESSFUL);

    }

    private SubOperationDescriptor<I, O> getSubOperationDescriptorByValue(I value) {
        try {
            return this.subOperationDescriptors.stream()
                    .filter(desc -> desc.predicate.test(value))
                    .collect(MoreCollectors.onlyElement());
        } catch (NoSuchElementException | IllegalArgumentException e) {
            throw new IllegalArgumentException(
                    "Каждому входящему элементу должна соответствовать одна под-операция", e);
        }
    }

    @SuppressWarnings("unused")
    public static SplitAndMergeOperationAggUntypedBuilder builder() {
        return new SplitAndMergeOperationAggUntypedBuilder(false);
    }

    public static SplitAndMergeOperationAggUntypedBuilder builderForPartialOperations() {
        return new SplitAndMergeOperationAggUntypedBuilder(true);
    }

    public static <S, R> SplitAndMergeOperationAggTypedBuilder<S, R> builderForPartialOperations(
            @SuppressWarnings("unused") Class<S> input, @SuppressWarnings("unused") Class<R> result) {
        return new SplitAndMergeOperationAggTypedBuilder<>(true);
    }

    private static class SubOperationDescriptor<S, R> {
        private final Predicate<S> predicate;
        private final OperationCreator<S, Operation<R>> operationCreator;
        private final int ordinal;  // порядок выполнения операций - чем меньше число, тем раньше выполняется.

        public SubOperationDescriptor(Predicate<S> predicate, OperationCreator<S, Operation<R>> operationCreator,
                                      int ordinal) {
            this.predicate = predicate;
            this.operationCreator = OperationCreators.createEmptyOperationOnEmptyInput(operationCreator);
            this.ordinal = ordinal;
        }

        Operation<R> createOperation(List<S> input) {
            return operationCreator.create(input);
        }
    }

    public static class SplitAndMergeOperationAggUntypedBuilder {
        private final boolean forPartialOperations;

        private SplitAndMergeOperationAggUntypedBuilder(boolean forPartialOperations) {
            this.forPartialOperations = forPartialOperations;
        }

        public <X, Y> SplitAndMergeOperationAggTypedBuilder<X, Y> addSubOperation(
                Predicate<X> predicate,
                OperationCreator<X, Operation<Y>> operationCreator) {
            SplitAndMergeOperationAggTypedBuilder<X, Y> builder =
                    new SplitAndMergeOperationAggTypedBuilder<>(forPartialOperations);
            builder.addSubOperation(predicate, operationCreator);
            return builder;
        }
    }

    public static class SplitAndMergeOperationAggTypedBuilder<S, R> {
        private List<SubOperationDescriptor<S, R>> subOperationDescriptors;
        private final boolean forPartialOperations;

        private SplitAndMergeOperationAggTypedBuilder(boolean forPartialOperations) {
            this.forPartialOperations = forPartialOperations;
            subOperationDescriptors = new ArrayList<>();
        }

        @SuppressWarnings("UnusedReturnValue")
        public SplitAndMergeOperationAggTypedBuilder<S, R> addSubOperation(
                Predicate<S> predicate,
                OperationCreator<S, Operation<R>> operationCreator) {
            subOperationDescriptors.add(new SubOperationDescriptor<>(predicate, operationCreator,
                    subOperationDescriptors.size()));
            return this;
        }

        public SplitAndMergeOperationAggregator<S, R> build() {
            return new SplitAndMergeOperationAggregator<>(subOperationDescriptors, forPartialOperations);
        }
    }

    /**
     * Собирает верхнеуровневый ValidationResult на основе ValidationResult'ов Result'ов MassResult'а
     */
    public static MassResult<Long> getMassResultWithAggregatedValidationResult(MassResult<Long> massResult) {
        //noinspection unchecked
        var validationResult = (ValidationResult<List<?>, Defect>) massResult.getValidationResult();
        ValidationResult<List<?>, Defect> mergedVr = new ValidationResult<>(validationResult.getValue());
        for (int i = 0; i < massResult.getResult().size(); i++) {
            ValidationResult<?, Defect> localVr = massResult.getResult().get(i).getValidationResult();
            if (null != localVr) {
                transferSubNodesWithIssues(localVr, mergedVr.getOrCreateSubValidationResult(index(i),
                        mergedVr.getValue().get(i)));
            }
        }
        return new MassResult<>(massResult.getResult(), mergedVr, massResult.getState());
    }
}
