package ru.yandex.direct.operation.creator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static ru.yandex.direct.operation.creator.OperationCreators.createEmptyPartiallyApplicableOperationOnEmptyInput;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.extractSubList;
import static ru.yandex.direct.utils.FunctionalUtils.intRange;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Декоратор {@link OperationCreator}, позволяет оперировать группами входящих значений. Группа это список элементов,
 * которые должны рассматриваься как одна логическая сущность. Т.е. если хотябы один элемент группы не валиден,
 * то невалидна вся группа. Применение операции к группе элементов, должно вызывать применение операции к каждому
 * элементу группы.
 * <p>
 * Оборачиваемый {@code flatOperationCreator} должен обеспечивать операцию над элементами составляющими группу.
 * <p>
 * При создании групповой операции, все элементы групп раскладываются в общий список и передаются в оборачиваемую
 * операцию. При этом осуществляется прозрачная трансляция ошибок валидации отдельных элементов на группу.
 * Результаты валидации элементов группы объединяются, чтобы соответствовать структуре исходной группы (списку).
 * <p>
 * Успешные результаты применения (например id) тоже объединятся в списки, соотвествующие структуре исходной группы.
 * <p>
 * Зачем: Этот создатель операции можно использовать, например, для добавления нескольких дополнительных таргетингов
 * к группе объявлений, так, что таргетинги к одной группе объявлений либо добавятся все, либо вернётся ошибка.
 *
 * @param <I> тип входящих в группу элементов операции
 * @param <O> тип элементов результата
 */
@ParametersAreNonnullByDefault
public class GroupOperationCreator<I, O> implements OperationCreator<List<I>, PartiallyApplicableOperation<List<O>>> {
    private final OperationCreator<I, PartiallyApplicableOperation<O>> flatOperationCreator;

    public GroupOperationCreator(OperationCreator<I, PartiallyApplicableOperation<O>> flatOperationCreator) {
        this.flatOperationCreator = createEmptyPartiallyApplicableOperationOnEmptyInput(flatOperationCreator);
    }

    @Override
    public PartiallyApplicableOperation<List<O>> create(List<List<I>> operationInput) {
        return new GroupOperation<>(flatOperationCreator, operationInput);
    }

    public static class GroupOperation<I, O> implements PartiallyApplicableOperation<List<O>> {
        private final List<I> flatInput;
        private final PartiallyApplicableOperation<O> flatOperation;
        private final Map<Integer, List<Integer>> originalToFlatIndexes;
        private final Map<Integer, Integer> flatToOriginalIndex;
        private final List<List<I>> originalOperationInput;

        GroupOperation(OperationCreator<I, PartiallyApplicableOperation<O>> flatOperationCreator,
                       List<List<I>> originalOperationInput) {
            FlattenedInput<I> flattenedInput = originalToFlatInput(originalOperationInput);

            this.flatInput = flattenedInput.getFlatInput();
            this.flatOperation = flatOperationCreator.create(flatInput);
            this.originalOperationInput = originalOperationInput;

            this.originalToFlatIndexes = flattenedInput.getOriginalToFlatIndexes();
            flatToOriginalIndex = EntryStream.of(originalToFlatIndexes)
                    .flatMapValues(Collection::stream)
                    .invert()
                    .toMap();
        }

        @Override
        public Set<Integer> getValidElementIndexes() {
            Set<Integer> flatIndexesOfInvalid = Sets.difference(
                    flatToOriginalIndex.keySet(), flatOperation.getValidElementIndexes());
            Set<Integer> originalIndexesOfInvalid = listToSet(flatIndexesOfInvalid, flatToOriginalIndex::get);
            return Sets.difference(originalToFlatIndexes.keySet(), originalIndexesOfInvalid);
        }

        @Override
        public MassResult<List<O>> apply(Set<Integer> originalIndexesToApply) {
            Set<Integer> flatIndexesToApply = StreamEx.of(originalIndexesToApply)
                    .map(originalToFlatIndexes::get)
                    .flatMap(Collection::stream)
                    .toSet();
            return makeResultForOriginal(flatOperation.apply(flatIndexesToApply));
        }

        @Override
        public MassResult<List<O>> prepareAndApply() {
            return prepare().orElse(apply(getValidElementIndexes()));
        }

        @Override
        public Optional<MassResult<List<O>>> prepare() {
            return flatOperation.prepare().map(this::makeResultForOriginal);
        }

        @Override
        public MassResult<List<O>> apply() {
            return apply(getValidElementIndexes());
        }

        @Override
        public MassResult<List<O>> cancel() {
            return makeResultForOriginal(flatOperation.cancel());
        }

        @Override
        public Optional<MassResult<List<O>>> getResult() {
            return flatOperation.getResult().map(this::makeResultForOriginal);
        }

        private MassResult<List<O>> makeResultForOriginal(MassResult<O> flatMassResult) {
            if (!flatMassResult.isSuccessful()) {
                // Если результат содержит ошибки уровня операции, то результатов выполнения операции нет
                return new MassResult<>(null, flatMassResult.getValidationResult(), flatMassResult.getState());
            }

            List<Result<O>> resultsForFlat = flatMassResult.getResult();
            checkState(resultsForFlat.size() == flatInput.size());

            List<Result<List<O>>> resultForOriginal = EntryStream.of(originalOperationInput)
                    .mapKeyValue((originalIndex, originalValue) ->
                            collectResult(originalValue,
                                    extractSubList(resultsForFlat,
                                            originalToFlatIndexes.getOrDefault(originalIndex, emptyList()))))
                    .toList();
            return new MassResult<>(resultForOriginal, ValidationResult.success(originalOperationInput),
                    flatMassResult.getState());
        }

        private Result<List<O>> collectResult(List<I> originalValue, List<Result<O>> results) {
            if (originalValue.isEmpty()) {
                checkState(results.isEmpty());
                return Result.successful(emptyList(), ValidationResult.success(originalValue));
            }
            Set<ResultState> resultStates = listToSet(results, Result::getState);
            if (resultStates.equals(singleton(ResultState.SUCCESSFUL))) {
                return new Result<>(mapList(results, Result::getResult),
                        collectValidationResult(originalValue, results), ResultState.SUCCESSFUL);
            } else if (resultStates.equals(singleton(ResultState.CANCELED))) {
                return Result.canceled(ValidationResult.success(originalValue));
            } else if (!resultStates.contains(ResultState.SUCCESSFUL)) {
                return Result.broken(collectValidationResult(originalValue, results));
            } else {
                throw new IllegalStateException();
            }
        }

        private ValidationResult<List<I>, Defect> collectValidationResult(List<I> originalValue,
                                                                          List<Result<O>> results) {
            ValidationResult<List<I>, Defect> validationResultForOriginal = new ValidationResult<>(originalValue);
            EntryStream.of(results).forKeyValue((indexInGroup, result) -> {
                I flatInputValue = originalValue.get(indexInGroup);
                ValidationResult<I, Defect> vr = validationResultForOriginal.getOrCreateSubValidationResult(
                        PathHelper.index(indexInGroup), flatInputValue);
                vr.merge(nvl(result.getValidationResult(), ValidationResult.success(flatInputValue)));
            });
            return validationResultForOriginal;
        }

    }

    private static class FlattenedInput<I> {
        private final List<I> flatInput;
        private final Map<Integer, List<Integer>> originalToFlatIndexes;

        FlattenedInput(List<I> flatInput, Map<Integer, List<Integer>> originalToFlatIndexes) {
            this.flatInput = flatInput;
            this.originalToFlatIndexes = originalToFlatIndexes;
        }

        List<I> getFlatInput() {
            return flatInput;
        }

        Map<Integer, List<Integer>> getOriginalToFlatIndexes() {
            return originalToFlatIndexes;
        }
    }

    private static <X> FlattenedInput<X> originalToFlatInput(List<List<X>> operationInput) {
        Map<Integer, List<Integer>> originalToFlatIndexes = new HashMap<>(operationInput.size());
        List<X> flatInput = new ArrayList<>();

        int flatIndex = 0;
        for (int originalIndex = 0; originalIndex < operationInput.size(); originalIndex++) {
            List<X> group = operationInput.get(originalIndex);
            flatInput.addAll(group);
            List<Integer> flatIndexes = intRange(flatIndex, flatIndex + group.size());
            originalToFlatIndexes.put(originalIndex, flatIndexes);
            flatIndex += flatIndexes.size();
        }
        return new FlattenedInput<>(flatInput, originalToFlatIndexes);
    }
}
