package ru.yandex.direct.operation.creator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;

import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.util.Deduplicator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.utils.converter.Converter;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.direct.validation.result.ValidationResult;


/**
 * Декоратор {@link OperationCreator}, который не отправляет в оборачиваемый {@code OperationCreator} дублирующиеся
 * входящие значения (только уникальные), но результат возвращается для всего входящего списка (который с дубликатами),
 * копирую результаты этих повторяющихся элементов.
 * <p>
 * Зачем: Обычно, операции валидируют входящий список элементов, не допуская дубликатов. В этом случае, для всех
 * одинаковых входящих элементов вернётся ошибка. Но в API, например, есть методы suspend/resume, в которые
 * можно передавать повторящиеся идентификаторы, и они должны успешно обработаться.
 * С помощью {@code IgnoreDuplicatesOperationCreator} можно спрятать от валидации операции дублирующиеся элементы.
 *
 * @param <I> тип входящих элементов операции
 * @param <O> тип элементов результата
 */
public class IgnoreDuplicatesOperationCreator<I, O> implements OperationCreator<I, Operation<O>> {
    private final OperationCreator<I, ? extends Operation<O>> originalOperationCreator;
    private final Comparator<I> elementComparator;
    private final Consumer<Result<O>> duplicatedResultVisitor;

    public IgnoreDuplicatesOperationCreator(OperationCreator<I, ? extends Operation<O>> originalOperationCreator,
                                            Comparator<I> elementComparator,
                                            Consumer<Result<O>> duplicatedResultVisitor) {
        this.originalOperationCreator = originalOperationCreator;
        this.elementComparator = elementComparator;
        this.duplicatedResultVisitor = duplicatedResultVisitor;
    }

    @Override
    public Operation<O> create(List<I> operationInput) {
        Deduplicator.Result<I> deduplicationResult = Deduplicator.deduplicate(operationInput, elementComparator);
        return new ConvertResultOperation<>(
                originalOperationCreator.create(deduplicationResult.getDeduplicated()),
                cutInDuplicates(operationInput.size(), deduplicationResult.getDeduplicationInfo(), operationInput));
    }

    /**
     * Вклейка дубликатов в результат.
     * <p>
     * Выполенена в виде конвертора, чтобы можно было использовать в ConvertResultOperation.
     */
    private Converter<MassResult<O>, MassResult<O>> cutInDuplicates(int originalListSize,
                                                                    List<List<Integer>> deduplicationInfo,
                                                                    List<I> operationInput) {
        return inputMassResult -> {
            if (!inputMassResult.isSuccessful()) {
                // Если результат содержит ошибки уровня операции, то результатов выполнения операции нет, значит
                // вклеивать ничего не надо.
                return inputMassResult;
            }
            Preconditions.checkState(inputMassResult.getResult().size() == deduplicationInfo.size(),
                    "Operation returns MassResult with unexpected size");

            List<Result<O>> resultWithDuplicates = new ArrayList<>(Collections.nCopies(originalListSize, null));
            EntryStream.zip(inputMassResult.getResult(), deduplicationInfo)
                    .forKeyValue((res, positions) -> positions.forEach(pos -> {
                        Result<O> copiedResult;
                        if (res.getValidationResult() == null) {
                            // Вообще ситуации когда validationResult == null и при этом статус не Successful
                            // быть не должно
                            ValidationResult<I, Defect> vr = res.getState() == ResultState.SUCCESSFUL
                                    ? ValidationResult.success(operationInput.get(pos))
                                    : ValidationResult.failed(operationInput.get(pos),
                                    new Defect<>(DefectIds.INVALID_VALUE));
                            copiedResult = new Result<>(res.getResult(), vr, res.getState());
                        } else {
                            copiedResult = new Result<>(res);
                        }
                        if (positions.size() > 1) {
                            // вызываем только для результатов, имеющих дубликаты
                            duplicatedResultVisitor.accept(copiedResult);
                        }
                        resultWithDuplicates.set(pos, copiedResult);
                    }));

            Preconditions.checkState(resultWithDuplicates.stream().noneMatch(Objects::isNull));
            return new MassResult<>(
                    resultWithDuplicates, inputMassResult.getValidationResult(), inputMassResult.getState());
        };
    }
}
