package ru.yandex.direct.operation.creator;

import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.operation.creator.OperationCreators.createEmptyPartiallyApplicableOperationOnEmptyInput;

/**
 * Декоратор {@link OperationCreator}, который отправляет в оборачиваемый {@code originalOperationCreator} только
 * некоторое подмножество входящих значений, которое задаётся индексами {@code subsetIndexes}.
 * <p>
 * Для входящих значений индексы которых не попали в {@code subsetIndexes}, оригинальная операция не вызывается,
 * а результат возвращается со статусом {@link ResultState#CANCELED}.
 * <p>
 * Значения попавшие в {@code subsetIndexes} отправятся в оригинальную операцию, которая вернёт для них
 * свой результат.
 * <p>
 * Результат операции создаваемой {@code OnlySubsetOperationCreator}, это результат для всего исходного списка входящих
 * значений, собранный так, что каждому элементу соотвествует свой результат: {@link ResultState#CANCELED}
 * или возвращённый из оборачиваемой операции.
 * <p>
 * Зачем: Этот создатель операции можно использовать, если, например, должны выполняться две операции совместно, после
 * {@link Operation#prepare()} первой стали известны невалидные элементы, вторую операция можно создать уже только
 * для валидных элементов, чтобы не выполнялась лишная работа по подготовке.
 *
 * @param <I> тип входящих элементов операции
 * @param <O> тип элементов результата
 */
public class OnlySubsetOperationCreator<I, O> implements OperationCreator<I, PartiallyApplicableOperation<O>> {
    private final OperationCreator<I, PartiallyApplicableOperation<O>> originalOperationCreator;
    private final Set<Integer> subsetIndexes;

    public OnlySubsetOperationCreator(OperationCreator<I, PartiallyApplicableOperation<O>> originalOperationCreator,
                                      Set<Integer> subsetIndexes) {
        this.originalOperationCreator = createEmptyPartiallyApplicableOperationOnEmptyInput(originalOperationCreator);
        this.subsetIndexes = subsetIndexes;
    }

    @Override
    public PartiallyApplicableOperation<O> create(List<I> originalInput) {
        return new OnlySubsetOperation(originalInput);
    }

    public class OnlySubsetOperation implements PartiallyApplicableOperation<O> {
        private final PartiallyApplicableOperation<O> operationOnSubset;
        private final List<I> originalInput;
        private final BiMap<Integer, Integer> subsetToOriginalIndex;

        OnlySubsetOperation(List<I> originalInput) {
            this.originalInput = originalInput;

            List<Integer> subsetIndexesList = StreamEx.of(subsetIndexes).sorted().toList();
            List<I> subsetInput = StreamEx.of(subsetIndexesList)
                    .map(originalInput::get)
                    .toList();

            Supplier<BiMap<Integer, Integer>> biMapSupplier = () -> HashBiMap.create(subsetIndexesList.size());
            subsetToOriginalIndex = EntryStream.of(subsetIndexesList)
                    .toCustomMap(biMapSupplier);

            this.operationOnSubset = originalOperationCreator.create(subsetInput);
        }

        @Override
        public Set<Integer> getValidElementIndexes() {
            return StreamEx.of(operationOnSubset.getValidElementIndexes())
                    .map(subsetToOriginalIndex::get)
                    .toImmutableSet();
        }

        @Override
        public MassResult<O> apply(Set<Integer> elementIndexesToApply) {
            BiMap<Integer, Integer> originalToSubsetIndex = subsetToOriginalIndex.inverse();
            Set<Integer> subsetInputIndexesToApply = StreamEx.of(elementIndexesToApply)
                    .map(originalToSubsetIndex::get)
                    .nonNull() // пропущенные индексы не должны применятся
                    .toImmutableSet();
            return cutInSkipped(operationOnSubset.apply(subsetInputIndexesToApply));
        }

        @Override
        public Optional<MassResult<O>> prepare() {
            return operationOnSubset.prepare().map(this::cutInSkipped);
        }

        @Override
        public MassResult<O> apply() {
            return cutInSkipped(operationOnSubset.apply());
        }

        @Override
        public MassResult<O> cancel() {
            return cutInSkipped(operationOnSubset.cancel());
        }

        @Override
        public Optional<MassResult<O>> getResult() {
            return operationOnSubset.getResult().map(this::cutInSkipped);
        }

        /**
         * Вклейка пропущенных элементов в результат.
         * <p>
         * Элементы вклеиваются как результаты с состоянием {@link ResultState#CANCELED}
         */
        private MassResult<O> cutInSkipped(MassResult<O> massResultForSubset) {
            if (!massResultForSubset.isSuccessful()) {
                // Если результат содержит ошибки уровня операции, то результатов выполнения операции нет, значит
                // вклеивать ничего не надо.
                return massResultForSubset;
            }
            Preconditions.checkState(massResultForSubset.getResult().size() == subsetIndexes.size(),
                    "Operation returns MassResult with unexpected size");

            Iterator<Result<O>> iteratorOfResults = massResultForSubset.getResult().iterator();

            List<Result<O>> resultWithSkipped = EntryStream.of(originalInput)
                    .mapKeyValue((idx, originalInputElement) -> {
                        if (subsetIndexes.contains(idx)) {
                            return iteratorOfResults.next();
                        } else {
                            return Result.<O>canceled(ValidationResult.success(originalInputElement));
                        }
                    })
                    .toList();

            return new MassResult<>(
                    resultWithSkipped, massResultForSubset.getValidationResult(), massResultForSubset.getState());
        }
    }
}
