package ru.yandex.direct.operation.creator;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;

import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.operation.creator.OperationCreators.createEmptyPartiallyApplicableOperationOnEmptyInput;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Декоратор {@link OperationCreator}, который позволяет добавлять валидатор {@code validator}
 * на оборачиваемую операцию, создаваемую посредством {@code operationCreator}. Валидатор применяется при создании
 * операции и ошибки из полученного результата валидации добавляются к результатам вызова оборачиваемой операции.
 * <p>
 * Если для любого из валидируемых элементов в пути объединенного результата валидации (полученного добавлением ошибок
 * из результата валидации валидатора) содержатся ошибки ({@link ValidationResult#hasAnyErrors()}) - результат для
 * этого элемента будет иметь статус {@link ResultState#BROKEN}, а итоговый результат операции
 * {@link ResultState#SUCCESSFUL}. Поэтому, если обернутая операция возвращает для одного из элементов результат
 * со статусом {@link ResultState#SUCCESSFUL}, но при этом результат валидации (так же полученный из операции)
 * данного элемента имеет ошибки - в итоговом результате для элемента будет статус {@link ResultState#BROKEN}.
 * <p>
 * Если операция вернула результат со статусом {@link ResultState#BROKEN} или объединенный результат валидации
 * содержит ошибки в корневом элементе ({@link ValidationResult#hasErrors()}) - возвращается итоговый результат
 * со статусом {@link ResultState#BROKEN}.
 *
 * @param <I> тип валидируемых элементов и передаваемых в оборачиваемую операцию
 * @param <O> тип элементов результата
 */
@ParametersAreNonnullByDefault
public class ValidationOperationCreator<I, O> implements OperationCreator<I, PartiallyApplicableOperation<O>> {

    private final OperationCreator<I, PartiallyApplicableOperation<O>> operationCreator;
    private final Validator<List<I>, Defect> validator;

    public ValidationOperationCreator(OperationCreator<I, PartiallyApplicableOperation<O>> operationCreator,
                                      Validator<List<I>, Defect> validator) {
        this.operationCreator = createEmptyPartiallyApplicableOperationOnEmptyInput(operationCreator);
        this.validator = validator;
    }

    @Override
    public PartiallyApplicableOperation<O> create(List<I> operationInput) {
        return new ValidationOperationDecorator<>(
                operationCreator.create(operationInput), validator.apply(operationInput), operationInput);
    }

    public static class ValidationOperationDecorator<I, O> implements PartiallyApplicableOperation<O> {

        private final PartiallyApplicableOperation<O> operation;
        private final ValidationResult<List<I>, Defect> validationResult;
        private final List<I> operationInput;

        private MassResult<O> result;

        ValidationOperationDecorator(PartiallyApplicableOperation<O> operation,
                                     ValidationResult<List<I>, Defect> validationResult,
                                     List<I> operationInput) {
            this.operation = operation;
            this.validationResult = validationResult;
            this.operationInput = operationInput;
        }

        @Override
        public Set<Integer> getValidElementIndexes() {
            if (validationResult.hasErrors()) {
                return Collections.emptySet();
            }

            return Sets.intersection(
                    operation.getValidElementIndexes(),
                    ValidationResult.getValidItemsWithIndex(validationResult).keySet());
        }

        @Override
        public MassResult<O> apply(Set<Integer> elementIndexesToApply) {
            checkApplyOrCancelPreconditions();

            var validIndexes = ValidationResult.getValidItemsWithIndex(validationResult).keySet();
            boolean indexesIsValid = validIndexes.containsAll(elementIndexesToApply);
            checkState(indexesIsValid, "indexes is not valid");

            result = mergeResults(operation.apply(elementIndexesToApply));
            return result;
        }

        @Override
        public Optional<MassResult<O>> prepare() {
            result = operation.prepare().orElse(null);

            if (validationResult.hasErrors()) {
                if (result == null) {
                    result = new MassResult<>(null, validationResult, ResultState.BROKEN);
                } else {
                    result = mergeResults(result);
                }
            }

            return Optional.ofNullable(result);
        }

        @Override
        public MassResult<O> apply() {
            return apply(getValidElementIndexes());
        }

        @Override
        public MassResult<O> cancel() {
            checkApplyOrCancelPreconditions();

            result = mergeResults(operation.cancel());
            return result;
        }

        @Override
        public Optional<MassResult<O>> getResult() {
            return Optional.ofNullable(result);
        }

        private void checkApplyOrCancelPreconditions() {
            checkState(result == null, "result is already computed");
        }

        private MassResult<O> mergeResults(MassResult<O> operationResult) {
            // Если в результате содержатся ошибки уровня операции - возвращается этот же результат,
            // без добавления результатов валидатора
            if (!operationResult.isSuccessful()) {
                return operationResult;
            }

            @SuppressWarnings("unchecked")
            var opValidationResult =
                    new ValidationResult<>((ValidationResult<List<?>, Defect>) operationResult.getValidationResult());

            // Некоторые операции (например GroupOperation) могут вернуть пустой список результатов валидаций,
            // поэтому нужно проставить их отдельно
            if (opValidationResult.getSubResults().isEmpty()) {
                for (int i = 0; i < operationInput.size(); i++) {
                    opValidationResult.addSubResult(index(i), operationResult.get(i).getValidationResult());
                }
            }

            ValidationResult.transferSubNodesWithIssues(validationResult, opValidationResult);
            var subResults = opValidationResult.getSubResults();

            // Если объедененный результат валидации имеет ошибки в корневом элементе - возвращается результат
            // со статусом BROKEN
            if (opValidationResult.hasErrors()) {
                return new MassResult<>(null, opValidationResult, ResultState.BROKEN);
            }

            List<O> resultItems = mapList(operationResult.getResult(), Result::getResult);

            var canceledElementIndexes = new HashSet<Integer>();
            for (int i = 0; i < operationInput.size(); i++) {
                boolean isCanceled = operationResult.get(i).getState() == ResultState.CANCELED;
                boolean isValid = !subResults.get(index(i)).hasAnyErrors();
                if (isCanceled && isValid) {
                    canceledElementIndexes.add(i);
                }
            }

            return MassResult.successfulMassAction(resultItems, opValidationResult, canceledElementIndexes);
        }
    }
}
