package ru.yandex.direct.operation.execution;

import java.util.Collection;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.result.MassResult;

/**
 * Стратегия выполнения, позволяющая выполнить группу операций,
 * если валидация каждой из них прошла успешно.
 */
@ParametersAreNonnullByDefault
public class AllOrNothingExecutionStrategy implements ExecutionStrategy<Operation<?>> {
    private boolean forcePrepareAll;
    private final Predicate<Optional<? extends MassResult<?>>> failCriteria;

    /**
     * @param forcePrepareAll когда одна из операций провалена, вызывать ли prepare у остальных
     */
    public AllOrNothingExecutionStrategy(boolean forcePrepareAll) {
        this(forcePrepareAll, false);
    }

    /**
     * Для операций, допускающих частичное выполнение, критерий успешности операции другой. Чтобы это учесть,
     * нужно выставить параметр forPartialOperations в true, для частичных операций
     */
    public AllOrNothingExecutionStrategy(boolean forcePrepareAll, boolean forPartialOperations) {
        this.forcePrepareAll = forcePrepareAll;
        failCriteria = forPartialOperations ? failForPartialYes : failForPartialNo;
    }

    /**
     * @return true, если все операции успешно выполнены, в противном случае false
     */
    @Override
    public boolean execute(Collection<? extends Operation<?>> operations) {
        if (prepare(operations)) {
            finalize(operations, Operation::apply);
            return true;
        }
        finalize(operations, Operation::cancel);
        return false;
    }

    public void validate(Collection<? extends Operation<?>> operations) {
        prepare(operations);
        finalize(operations, Operation::cancel);
    }

    /**
     * Выполнить prepare() для всех операций
     * @param operations операции для выполнения prepare()
     * @return true если prepare() выполнено успешно для всех операций и можно вызывать apply(), иначе false
     */
    private boolean prepare(Collection<? extends Operation<?>> operations) {
        boolean prepareFailed = false;
        for (Operation<?> operation : operations) {
            Optional<? extends MassResult<?>> resultOfPrepare = operation.prepare();
            if (failCriteria.test(resultOfPrepare)) {
                prepareFailed = true;
                if (!forcePrepareAll) {
                    break;
                }
            }
        }
        return !prepareFailed;
    }

    private void finalize(Collection<? extends Operation<?>> operations,
                          Consumer<Operation<?>> finalizeMethod) {
        operations.stream()
                // запускаем операции последовательно, чтобы сохранить порядок обработки операций
                .sequential()
                // операции с готовыми результатами могут быть, если происходит работа с операциями, допускающими
                // частичное выполнение, когда часть данных в операции данные невалидна, но в целом результат
                // успешен (нет ошибки уровня операции)
                .filter(op -> op.getResult().isEmpty())
                .forEach(finalizeMethod);
    }

    // Частичная операция считается упавшей, только в том случае, если prepare вернул результат и этот
    // результат не успешный
    private static Predicate<Optional<? extends MassResult<?>>> failForPartialYes =
            resultOfPrepare -> resultOfPrepare.map(r -> !r.isSuccessful()).orElse(false);

    // Нечастичная операция считается упавшей в случае, если prepare вернул любой результат
    private static Predicate<Optional<? extends MassResult<?>>> failForPartialNo = Optional::isPresent;
}
