package ru.yandex.direct.operation.operationwithid;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItemsWithIndex;
import static ru.yandex.direct.validation.util.ValidationUtils.cloneValidationResultSubNodesWithIssues;

/**
 * Пошаговая операция удаления/изменения объектов.
 * Позволяет удалять или изменять свойства объектов, если не требуется {@link AppliedChanges}
 * <p>
 * Операция разбита на несколько этапов, часть из которых можно переопределить в классе-потомке
 * 1. провалидировать идентификаторы удаляемых/изменяемых объектов (обязательно для определения)
 * 2. удалить/изменить объекты из базы (обязательно для определения)
 * 3. выполнить дополнительные действия до или после удаления/изменения (опционально переопределяемо в потомке)
 */
public abstract class AbstractOperationWithId implements Operation<Long> {

    private final Applicability applicability;
    private final List<Long> modelIds;

    private ValidationResult<List<Long>, Defect> validationResult;
    private Map<Integer, Long> validModelIds;

    private boolean prepared;
    private boolean executed;
    private MassResult<Long> result;

    public AbstractOperationWithId(Applicability applicability, List<Long> modelIds) {
        this.applicability = applicability;
        this.modelIds = ImmutableList.copyOf(modelIds);
    }

    @Override
    public final Optional<MassResult<Long>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        validateInternal();
        if (result != null) {
            return Optional.of(result);
        }

        return Optional.empty();
    }

    @Override
    public final MassResult<Long> apply() {
        checkApplyOrCancelPreconditions();
        applyInternal();
        return result;
    }

    @Override
    public final MassResult<Long> cancel() {
        checkApplyOrCancelPreconditions();
        cancelInternal();
        return result;
    }

    @Override
    public final Optional<MassResult<Long>> getResult() {
        return Optional.ofNullable(result);
    }

    protected boolean isPrepared() {
        return prepared;
    }

    protected boolean isExecuted() {
        return executed;
    }

    /**
     * Внутренний метод валидации идентификаторов.
     * Вызывает пользовательское определение валидации идентификаторов,
     * если валидных элементов нет, то сразу задает результат операции.
     */
    private void validateInternal() {
        validationResult = validate(modelIds);
        validationResult = cloneValidationResultSubNodesWithIssues(validationResult);
        validModelIds = getValidItemsWithIndex(validationResult);

        if (validationResult.hasErrors() || validModelIds.isEmpty()
                || (isFull(applicability) && validationResult.hasAnyErrors())) {
            List<Long> ids = validationResult.getValue();
            this.result = MassResult.brokenMassAction(ids, validationResult);
        }
    }

    /**
     * Внутренний метод применения изменений.
     * Выставляется флаг выполнения операции и для валидных элементов по переданным индексам
     * выполняется операция вместе с вспомогательными методами до и после удаления.
     * <p>
     * Создается конечный успешный результат операции.
     */
    private void applyInternal() {
        executed = true;
        List<Long> validIds = getValidItems(validationResult);

        beforeExecution(validIds);
        execute(validIds);
        List<Long> ids = validationResult.getValue();
        result = MassResult.successfulMassAction(ids, validationResult);
        afterExecution(validIds);
    }

    private void cancelInternal() {
        executed = true;
        List<Long> results = Collections.nCopies(modelIds.size(), null);
        // все валидные элементы помечаем отменёнными, т.к. для них не выполнялась операция
        result = MassResult.successfulMassAction(results, validationResult, validModelIds.keySet());
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
    }

    /**
     * Абстрактный метод валидации идентификаторов.
     *
     * @param ids список идентификаторов, который нужно провалидировать
     * @return результат валидации списка идентификаторов.
     */
    protected abstract ValidationResult<List<Long>, Defect> validate(List<Long> ids);

    /**
     * Выполняется непосредственно перед удалением. Метод можно реализовать в потомке для
     * выполнения дополнительных действий.
     */
    @SuppressWarnings("unused")
    protected void beforeExecution(List<Long> ids) {
    }

    /**
     * Абстрактный метод удаления объектов.
     */
    protected abstract void execute(List<Long> ids);

    /**
     * Выполняется непосредственно после удаления в методе {@code execute()}.
     * Метод можно реализовать в потомке для выполнения дополнительных действий после удаления.
     */
    @SuppressWarnings("unused")
    protected void afterExecution(List<Long> ids) {
    }
}
