package ru.yandex.direct.operation.tree;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.getOrCreateSubValidationResult;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidation;

public class ItemSubOperationExecutor<P, C, SO extends SubOperation<C>> {

    private final SO subOperation;
    private final ImmutableMap<Integer, Integer> indexMap;
    private final ImmutableMap<Integer, C> parentIndexToChildMap;
    private final String parentToChildrenPropertyName;

    private ItemSubOperationExecutor(
            SO subOperation,
            Map<Integer, Integer> indexMap,
            Map<Integer, C> parentIndexToChildMap,
            String parentToChildPropertyName) {
        this.subOperation = subOperation;
        this.indexMap = ImmutableMap.copyOf(indexMap);
        this.parentIndexToChildMap = ImmutableMap.copyOf(parentIndexToChildMap);
        this.parentToChildrenPropertyName = parentToChildPropertyName;
    }

    public static ItemSubOperationExecutorBuilder builder() {
        return new ItemSubOperationExecutorBuilder();
    }

    protected static <FP extends Model, P, FC, C, SO extends SubOperation<C>> ItemSubOperationExecutor<P, C, SO>
    create(
            List<FP> fakeParents,
            ModelProperty<? super FP, FC> fakeChildProperty,
            Function<FC, C> fakeChildToRealChildFn,
            Predicate<FP> fakeParentFilter,
            SubOperationCreator<FC, SO> operationCreator) {
        Map<Integer, Integer> indexMap = new HashMap<>();
        Predicate<FP> predicate = fakeParentFilter.and(fakeParent -> fakeChildProperty.get(fakeParent) != null);
        List<FC> fakeChildren = extractSubList(indexMap, fakeParents, predicate, fakeChildProperty::get);
        SO subOperation = operationCreator.create(fakeChildren);

        Map<Integer, C> parentIndexToChildMap =
                getParentIndexToChildMap(fakeParents, predicate, fakeChildProperty, fakeChildToRealChildFn);

        return new ItemSubOperationExecutor<>(subOperation, indexMap,
                parentIndexToChildMap, fakeChildProperty.name());
    }

    public SO getSubOperation() {
        return subOperation;
    }

    public Map<Integer, Integer> getIndexMap() {
        return indexMap;
    }

    public void prepare(ValidationResult<List<P>, Defect> parentsResult) {
        ValidationResult<List<C>, Defect> flatChildrenResults = subOperation.prepare();
        mergeSubItemValidationResults(parentsResult, flatChildrenResults, indexMap, parentIndexToChildMap,
                parentToChildrenPropertyName);
    }

    public void apply() {
        subOperation.apply();
    }

    /**
     * Объекты из исходного списка (list), которые удовлетворяют условию (predicate) добавить в отдельный список
     * (subList).
     * Сохранить в мапу (sourceToDestIndexMap) соответствие индексов из list к subList.
     */
    public static <S, D> List<D> extractSubList(Map<Integer, Integer> sourceToDestIndexMap,
                                                List<S> list, Predicate<S> predicate, Function<S, D> converter) {
        List<D> subList = new ArrayList<>(list.size());
        int destIndex = 0;
        for (int sourceIndex = 0; sourceIndex < list.size(); sourceIndex++) {
            S item = list.get(sourceIndex);
            if (predicate.test(item)) {
                subList.add(converter.apply(item));
                sourceToDestIndexMap.put(sourceIndex, destIndex++);
            }
        }
        return subList;
    }

    public static <T> List<T> extractSubList(Map<Integer, Integer> sourceToDestIndexMap,
                                             List<T> list, Predicate<T> predicate) {
        return extractSubList(sourceToDestIndexMap, list, predicate, Function.identity());
    }

    public static <P, C> void mergeSubItemValidationResults(
            ValidationResult<List<P>, Defect> parentsResult,
            ValidationResult<List<C>, Defect> flatChildrenResult,
            Map<Integer, Integer> parentToChildIndexMap,
            Map<Integer, C> parentIndexToChildMap,
            String parentToChildPropertyName) {
        checkArgument(parentsResult != null, "validation result of parents list is required");
        checkArgument(flatChildrenResult != null, "validation result of flat children list is required");
        checkArgument(parentToChildIndexMap != null, "parent-to-children index map is required");
        checkArgument(parentIndexToChildMap != null, "parentIndex-to-child map is required");
        checkArgument(parentToChildPropertyName != null,
                "name of the parent property which contains children list is required");
        checkArgument(parentIndexToChildMap.keySet().equals(parentToChildIndexMap.keySet()),
                "key sets of parentToChildIndexMap and parentIndexToChildMap must be equal");

        List<P> parents = parentsResult.getValue();
        parentToChildIndexMap.forEach((parentIndex, flatChildIndex) -> {

            ValidationResult<?, Defect> parentResult =
                    getOrCreateSubValidationResult(parentsResult, index(parentIndex), parents.get(parentIndex));

            ValidationResult<?, Defect> sourceChildResult =
                    flatChildrenResult.getSubResults().get(index(flatChildIndex));
            if (sourceChildResult == null) {
                return;
            }

            ValidationResult<?, Defect> destChildResult =
                    getOrCreateSubValidationResult(parentResult, field(parentToChildPropertyName),
                            sourceChildResult.getValue());

            transferIssuesFromValidationToValidation(sourceChildResult, destChildResult);
        });
    }

    private static <FP extends Model, FC, C> Map<Integer, C> getParentIndexToChildMap(List<FP> fakeParents,
                                                                                      Predicate<FP> fakeParentFilter, ModelProperty<? super FP, FC> parentToFakeChildProperty,
                                                                                      Function<FC, C> fakeChildToChildFn) {
        Map<Integer, C> parentIndexToChildMap = new HashMap<>();
        for (int i = 0; i < fakeParents.size(); i++) {
            FP fakeParent = fakeParents.get(i);
            if (!fakeParentFilter.test(fakeParent)) {
                continue;
            }
            FC fakeChild = parentToFakeChildProperty.get(fakeParents.get(i));
            parentIndexToChildMap.put(i, fakeChildToChildFn.apply(fakeChild));
        }
        return parentIndexToChildMap;
    }
}
