package ru.yandex.direct.operation.tree;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.MultimapBuilder;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.getOrCreateSubValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.intRange;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidationWithNewValue;

public class ListSubOperationExecutor<P, C, SO extends SubOperation<C>> {

    private final SO subOperation;
    private final ListMultimap<Integer, Integer> indexMultimap;
    private final ListMultimap<Integer, C> parentIndexToChildrenMultimap;
    private final String parentToChildrenPropertyName;

    private ListSubOperationExecutor(SO subOperation,
                                     ListMultimap<Integer, Integer> indexMultimap,
                                     ListMultimap<Integer, C> parentIndexToChildrenMultimap,
                                     String parentToChildrenPropertyName) {
        this.subOperation = subOperation;
        this.indexMultimap = ImmutableListMultimap.copyOf(indexMultimap);
        this.parentIndexToChildrenMultimap = ImmutableListMultimap.copyOf(parentIndexToChildrenMultimap);
        this.parentToChildrenPropertyName = parentToChildrenPropertyName;
    }

    public static ListSubOperationExecutorBuilder builder() {
        return new ListSubOperationExecutorBuilder();
    }

    protected static <FP extends Model, P, FC extends Model, C, SO extends SubOperation<C>>
    ListSubOperationExecutor<P, C, SO> create(
            List<FP> fakeParents,
            ModelProperty<? super FP, List<FC>> fakeChildrenProperty,
            Function<FC, C> fakeChildToRealChildFn,
            SubOperationCreator<FC, SO> operationCreator) {
        ListMultimap<Integer, Integer> indexMultimap = MultimapBuilder.hashKeys().arrayListValues().build();
        ListMultimap<Integer, C> parentIndexToChildrenMultimap =
                getParentIndexToChildrenMap(fakeParents, fakeChildrenProperty, fakeChildToRealChildFn);
        List<FC> fakeChildren =
                extractChildrenSubListsToFlatList(indexMultimap, fakeParents, fakeChildrenProperty::get);
        SO subOperation = operationCreator.create(fakeChildren);
        return new ListSubOperationExecutor<>(subOperation, indexMultimap,
                parentIndexToChildrenMultimap, fakeChildrenProperty.name());
    }

    public SO getSubOperation() {
        return subOperation;
    }

    public ListMultimap<Integer, Integer> getIndexMultimap() {
        return ImmutableListMultimap.copyOf(indexMultimap);
    }

    public void prepare(ValidationResult<List<P>, Defect> parentsResult) {
        ValidationResult<List<C>, Defect> flatChildrenResults = subOperation.prepare();
        mergeChildrenSubListValidationResults(parentsResult, flatChildrenResults, indexMultimap,
                parentIndexToChildrenMultimap, parentToChildrenPropertyName);
    }

    public void apply() {
        subOperation.apply();
    }

    /**
     * Здесь "родительский" объект - это объект, содержащий список объектов,
     * называемых "дочерними".
     * <p>
     * Из списка родительских объектов извлекает списки дочерних в один плоский список,
     * при этом заполняя мультимапу индексами для последующего восстановления связи
     * между плоским списком и исходными списками. Ключ в мультимапе - это индекс
     * родительского объекта, а значение - список индексов в плоском списке в том же
     * порядке, что и в исходном.
     *
     * @param indexMap  мультимапа индексов, заполняемая в процессе для сохранения
     *                  связи между исходными списками и плоским списком.
     * @param parents   список родительских объектов, содержащие списки дочерних объектов
     * @param extractor функция, извлекающая список дочерних объектов из родительского
     * @param <P>       тип родительского объекта
     * @param <C>       тип дочернего объекта
     * @return плоский список дочерних объектов
     */
    public static <P, C> List<C> extractChildrenSubListsToFlatList(ListMultimap<Integer, Integer> indexMap,
                                                                   List<P> parents, Function<P, List<C>> extractor) {
        indexMap.clear();
        List<C> allChildren = new ArrayList<>();
        int childrenFlatIndex = 0;

        for (int parentIndex = 0; parentIndex < parents.size(); parentIndex++) {
            P parent = parents.get(parentIndex);

            List<C> currentChildren = extractor.apply(parent);
            if (isEmpty(currentChildren)) {
                continue;
            }

            allChildren.addAll(currentChildren);

            List<Integer> flatIndexes = intRange(childrenFlatIndex, childrenFlatIndex + currentChildren.size());
            indexMap.putAll(parentIndex, flatIndexes);
            childrenFlatIndex += flatIndexes.size();
        }
        return allChildren;
    }

    /**
     * Контекст. Есть список "родительских" объектов, каждый из которых содержит список
     * "дочерних объектов". Родительские объекты валидируются отдельно и в результате
     * получается их результат валидации. Дочерние объекты извлекаются из родительских
     * в плоский список и валидируются отдельно. При извлечении в плоский список запоминается
     * связь между индексами дочерних объектов внутри родительских и их индексами в плоском списке
     * в виде мультимапы, ключи которой соответствуют индексам родительских объектов, а значения
     * соответствуют списку индексов ее дочерних объектов в плоском списке в том же порядке, что и
     * в родительском объекте. Например, "3 -> [10, 11]" означает, что 0-й и 1-й дочерние объекты
     * 3-го родительского объекта лежат в плоском списке под индексами 10 и 11.
     * <p>
     * Данный метод подмерживает результаты валидации дочерних объектов к соответствующим результатам
     * валидации родительских объектов таким образом, чтобы это выглядело так, будто дочерние объекты
     * валидировались прямо внутри родительских.
     *
     * @param parentsResult                результат валидации списка родительских объектов
     * @param flatChildrenResult           результатов валидации плоского списка дочерних объектов
     * @param parentToChildrenIndexMap     мапа индексов родительских объектов на индексы дочерних в плоском списке
     * @param parentIndexToChildrenMap     мапа индексов родительских объектов на списки их дочерних объектов
     * @param parentToChildrenPropertyName имя свойства, по которому в родительских объектах лежат дочерние
     * @param <P>                          тип родительского объекта
     * @param <C>                          тип дочернего объекта
     */
    public static <P, C> void mergeChildrenSubListValidationResults(
            ValidationResult<List<P>, Defect> parentsResult,
            ValidationResult<List<C>, Defect> flatChildrenResult,
            ListMultimap<Integer, Integer> parentToChildrenIndexMap,
            ListMultimap<Integer, C> parentIndexToChildrenMap,
            String parentToChildrenPropertyName) {
        checkArgument(parentsResult != null, "validation result of parents list is required");
        checkArgument(flatChildrenResult != null, "validation result of flat children list is required");
        checkArgument(parentToChildrenIndexMap != null, "parent-to-children index map is required");
        checkArgument(parentIndexToChildrenMap != null, "parentIndex-to-children map is required");
        checkArgument(parentToChildrenPropertyName != null,
                "name of the parent property which contains children list is required");
        checkArgument(parentIndexToChildrenMap.keySet().equals(parentToChildrenIndexMap.keySet()),
                "key sets of parentToChildrenIndexMap and parentIndexToChildrenMap must be equal");

        parentToChildrenIndexMap.asMap().forEach((parentIndex, flatChildrenIndexes) -> {
            Collection<C> children = parentIndexToChildrenMap.get(parentIndex);
            checkArgument(flatChildrenIndexes.size() == children.size(),
                    "parentToChildrenIndexMap.get(%s).size() != parentIndexToChildrenMap.get(%s).size()",
                    flatChildrenIndexes.size(), children.size());
        });

        List<P> parents = parentsResult.getValue();
        for (Integer parentIndex : parentToChildrenIndexMap.keySet()) {

            List<Integer> childrenFlatIndexes = parentToChildrenIndexMap.get(parentIndex);
            if (childrenFlatIndexes.isEmpty()) {
                continue;
            }

            ValidationResult<?, Defect> parentResult =
                    getOrCreateSubValidationResult(parentsResult, index(parentIndex), parents.get(parentIndex));

            for (int childIndex = 0; childIndex < childrenFlatIndexes.size(); childIndex++) {
                int childFlatIndex = childrenFlatIndexes.get(childIndex);

                ValidationResult<?, Defect> sourceChildResult =
                        flatChildrenResult.getSubResults().get(index(childFlatIndex));
                if (sourceChildResult == null) {
                    continue;
                }

                ValidationResult<?, Defect> destChildrenResult =
                        getOrCreateSubValidationResult(parentResult, field(parentToChildrenPropertyName),
                                parentIndexToChildrenMap.get(parentIndex));

                ValidationResult<?, Defect> destChildResult =
                        getOrCreateSubValidationResult(destChildrenResult, index(childIndex),
                                sourceChildResult.getValue());

                transferIssuesFromValidationToValidationWithNewValue(sourceChildResult, destChildResult);
            }
        }
    }

    private static <T extends Model, F, C> ListMultimap<Integer, C> getParentIndexToChildrenMap(List<T> parents,
                                                                                                ModelProperty<? super T, List<F>> parentToFakeChildrenProperty, Function<F, C> fakeChildToChildFn) {
        ListMultimap<Integer, C> parentIndexToChildrenProperty =
                MultimapBuilder.hashKeys().arrayListValues().build();
        for (int i = 0; i < parents.size(); i++) {
            List<F> fakeChildren = parentToFakeChildrenProperty.get(parents.get(i));
            if (!isEmpty(fakeChildren)) {
                parentIndexToChildrenProperty.putAll(i, mapList(fakeChildren, fakeChildToChildFn));
            }
        }
        return parentIndexToChildrenProperty;
    }
}
