package ru.yandex.direct.operation.tree;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.base.Functions;
import one.util.streamex.EntryStream;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.PartiallyApplicableModelOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.safeCast;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidationWithNewValue;

public class TreeOperationUtils {

    private TreeOperationUtils() {
    }

    public static <T> void mergeSubListValidationResults(ValidationResult<List<T>, Defect> destListVr,
                                                         @Nullable ValidationResult<List<T>, Defect> sourceListVr,
                                                         Map<Integer, Integer> destToSourceIndexMap) {
        mergeSubListValidationResultsFromAnotherObject(destListVr, sourceListVr, destToSourceIndexMap);
    }

    public static <T, S> void mergeSubListValidationResultsFromAnotherObject(ValidationResult<List<T>, Defect> destListVr,
                                                                             @Nullable ValidationResult<List<S>, Defect> sourceListVr,
                                                                             Map<Integer, Integer> destToSourceIndexMap) {
        if (sourceListVr == null) {
            return;
        }
        checkArgument(destListVr.getValue() != null, "null list can not contain any validation results");
        checkArgument(sourceListVr.getValue() != null, "source validation result can not contain null as value");
        int maxDestIndex = destToSourceIndexMap.keySet().stream().mapToInt(Integer::valueOf).max().orElse(0);
        checkArgument(maxDestIndex <= destListVr.getValue().size(),
                "max destination validation result index is greater than its list size");
        int maxSourceIndex = destToSourceIndexMap.values().stream().mapToInt(Integer::valueOf).max().orElse(0);
        checkArgument(maxSourceIndex <= sourceListVr.getValue().size(),
                "max source validation result index is greater than its list size");

        sourceListVr.getErrors().forEach(destListVr::addError);
        sourceListVr.getWarnings().forEach(destListVr::addWarning);

        destToSourceIndexMap.forEach((destIndex, sourceIndex) -> {
            ValidationResult<?, Defect> sourceVr =
                    sourceListVr.getSubResults().get(index(sourceIndex));
            if (sourceVr != null) {
                ValidationResult<?, Defect> destVr =
                        getOrCreateSubValidationResult(destListVr, index(destIndex), sourceVr.getValue());
                transferIssuesFromValidationToValidationWithNewValue(sourceVr, destVr);
            }
        });
    }

    public static <T, D> ValidationResult<?, D> getOrCreateSubValidationResult(
            ValidationResult<T, D> validationResult, PathNode pathNode, Object replacement) {
        ValidationResult<?, D> subResult = validationResult.getSubResults().get(pathNode);
        if (subResult == null) {
            subResult = validationResult.getOrCreateSubValidationResult(pathNode, replacement);
        }
        return subResult;
    }

    public static <T extends ModelWithId> ValidationResult<List<T>, Defect> preparePartialModelOperationAndGetValidationResult(
            @Nullable PartiallyApplicableModelOperation<? super T, ?> operation, List<T> inputObjects) {
        ValidationResult<List<T>, Defect> vr = new ValidationResult<>(inputObjects);
        if (operation != null) {
            ValidationResult<?, Defect> operationVr = operation.prepare()
                    .<ValidationResult<?, Defect>>map(MassResult::getValidationResult)
                    .orElseGet(operation::getValidationResult);
            vr.getErrors().addAll(operationVr.getErrors());
            vr.getWarnings().addAll(operationVr.getWarnings());
            operationVr.getSubResults().forEach((pathNode, subResult) -> {
                if (subResult.hasAnyErrors() || subResult.hasAnyWarnings()) {
                    Object value = ifNotNullOrDefault(safeCast(pathNode, PathNode.Index.class),
                            Functions.compose(inputObjects::get, PathNode.Index::getIndex), subResult.getValue());
                    ValidationResult<?, Defect> newSubResult = vr.getOrCreateSubValidationResult(pathNode, value);
                    transferSubNodesWithIssues(subResult, newSubResult);
                }
            });
        }
        return vr;
    }

    @SuppressWarnings("unchecked")
    public static <T> ValidationResult<List<T>, Defect> prepareNullableOperationAndGetValidationResult(
            @Nullable Operation<?> operation, List<T> inputObjects) {
        return (ValidationResult<List<T>, Defect>)
                Optional.ofNullable(operation)
                        .flatMap(Operation::prepare)
                        .map(MassResult::getValidationResult)
                        .orElseGet(() -> new ValidationResult(inputObjects));
    }

    @SuppressWarnings("unchecked")
    public static <T> ValidationResult<List<T>, Defect> applyNullableOperationAndGetValidationResult(
            @Nullable Operation<?> operation, List<T> inputObjects) {
        if (operation == null) {
            return new ValidationResult(inputObjects);
        }

        if (operation.getResult().isPresent()) {
            return (ValidationResult<List<T>, Defect>) operation.getResult()
                    .get()
                    .getValidationResult();
        }

        return (ValidationResult<List<T>, Defect>) operation
                .apply()
                .getValidationResult();
    }

    @Nullable
    public static <T> MassResult<T> cancelNullableOperationAndGetMassResult(
            @Nullable Operation<T> operation) {
        if (operation == null) {
            return null;
        }

        if (operation.getResult().isPresent()) {
            return operation.getResult().get();
        }
        return operation
                .cancel();
    }

    public static Set<Integer> computeCanceledIndexes(@Nullable MassResult<Long> oldMassResult,
                                                      @Nullable MassResult<Long> newMassResult,
                                                      Map<Integer, Integer> indexToOldOperationIndexMap,
                                                      Map<Integer, Integer> indexToNewOperationIndexMap) {
        Set<Integer> oldIndexCanceled = Optional.ofNullable(oldMassResult)
                .map(MassResult::getResult)
                .map(r -> EntryStream.of(r)
                        .filterValues(res -> res.getState() == ResultState.CANCELED)
                        .keys()
                        .toSet())
                .orElseGet(Collections::emptySet);
        Set<Integer> newIndexCanceled = Optional.ofNullable(newMassResult)
                .map(MassResult::getResult)
                .map(r -> EntryStream.of(r)
                        .filterValues(res -> res.getState() == ResultState.CANCELED)
                        .keys()
                        .toSet())
                .orElseGet(Collections::emptySet);

        Set<Integer> canceledIndexes = new HashSet<>(EntryStream.of(indexToOldOperationIndexMap)
                .filterValues(oldIndexCanceled::contains)
                .keys()
                .toSet());
        canceledIndexes.addAll(EntryStream.of(indexToNewOperationIndexMap)
                .filterValues(newIndexCanceled::contains)
                .keys()
                .toSet());
        return canceledIndexes;
    }

}
