package ru.yandex.direct.operation.update;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ContiguousSet;
import com.google.common.collect.DiscreteDomain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Range;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.PartiallyApplicableModelOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.SetsDiff;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItemsWithIndex;
import static ru.yandex.direct.validation.util.ValidationUtils.cloneValidationResultSubNodesWithIssues;
import static ru.yandex.direct.validation.util.ValidationUtils.extractOnlyValidAppliedChangesWithIndex;
import static ru.yandex.direct.validation.util.ValidationUtils.modelChangesValidationToModelValidation;

/**
 * Класс представляет собой абстрактную реализацию операции обновления.
 * Операция состоит из следующих шагов:
 * 1. провалидировать список ModelChanges;
 * 2. загрузить модели для валидных ModelChanges;
 * 3. применить валидные ModelChanges к соответствующим загруженным моделям;
 * 4. преобразовать результат валидации ModelChanges к результату валидации моделей;
 * 5. провалидировать модели;
 * 6. сохранить изменения;
 * 7. выполнить дополнительный действия (обновить статусы и т.п.).
 * <p>
 * Чтобы реализовать операцию обновления конкретного типа моделей нужно отнаследоваться
 * от данного класса и реализовать необходимые методы. Данный класс содержит как
 * обязательные для реализации абстрактные методы, так и пустые дефолтные методы-коллбэки
 * для опционального подключения к разным этапам выполнения операции.
 *
 * @param <M> тип модели
 */
public abstract class AbstractUpdateOperation<M extends ModelWithId, R> implements PartiallyApplicableModelOperation<M, R> {

    private final Applicability applicability;
    private final List<ModelChanges<M>> modelChanges;
    private final Function<Long, M> modelStubCreator;
    private final Set<ModelProperty<?, ?>> sensitiveProperties;
    private final Map<ModelProperty<? super M, ?>, BiFunction<Object, Object, Boolean>> customModelEquals;
    protected ValidationResult<List<M>, Defect> validationResult;
    // доступно после валидации ModelChanges
    private ValidationResult<List<ModelChanges<M>>, Defect> modelChangesValidationResult;
    private Map<Integer, ModelChanges<M>> validModelChanges; // валидные элементы вместе с их индексами
    // доступно после загрузки моделей и применения изменений к ним
    private Map<Integer, AppliedChanges<M>> appliedChangesForValidModelChanges;
    private Map<Integer, AppliedChanges<M>> validAppliedChanges;

    //доступно при применении изменений
    private Map<Integer, AppliedChanges<M>> applicableAppliedChanges;

    private boolean prepared;
    private boolean executed;
    private MassResult<R> result;

    public AbstractUpdateOperation(Applicability applicability, List<ModelChanges<M>> modelChanges,
                                   Function<Long, M> modelStubCreator) {
        this(applicability, modelChanges, modelStubCreator, emptySet());
    }

    public AbstractUpdateOperation(Applicability applicability, List<ModelChanges<M>> modelChanges,
                                   Function<Long, M> modelStubCreator,
                                   Set<ModelProperty<?, ?>> sensitiveProperties) {
        this(applicability, modelChanges, modelStubCreator, sensitiveProperties, emptyMap());
    }

    public AbstractUpdateOperation(
            Applicability applicability,
            List<ModelChanges<M>> modelChanges,
            Function<Long, M> modelStubCreator,
            Set<ModelProperty<?, ?>> sensitiveProperties,
            Map<ModelProperty<? super M, ?>,
            BiFunction<Object, Object, Boolean>> customModelEquals) {
        this.applicability = applicability;
        this.modelChanges = ImmutableList.copyOf(modelChanges);
        this.modelStubCreator = modelStubCreator;
        this.sensitiveProperties = sensitiveProperties;
        this.customModelEquals = customModelEquals;
    }

    private static <M extends ModelWithId> void checkModelsCollectionsMatch(
            Collection<M> modelsCollection,
            Collection<ModelChanges<M>> modelChangesCollection) {
        Set<Long> modelIds = StreamEx.of(modelsCollection)
                .map(ModelWithId::getId)
                .toSet();
        Set<Long> modelChangesIds = StreamEx.of(modelChangesCollection)
                .map(ModelChanges::getId)
                .toSet();
        SetsDiff<Long> diff = SetsDiff.createDiff(modelIds, modelChangesIds, Comparator.naturalOrder());
        checkState(diff.isEmpty(),
                String.format("set of models ids is not equal to set of ModelChanges ids: %s", diff));
    }

    /**
     * В этом методе проводятся два этапа валидации и могут порождаются артефакты,
     * необходимые для выполнения обновления. Само обновление не производится.
     * <p>
     * Метод необходимо вызвать перед вызовом {@link #apply()}.
     * <p>
     * Метод можно выполнить только один раз.
     *
     * @return отрицательный результат (элементы MassResult содержат null'ы), если операция провалена на одном из
     * этапов валидации,
     * в противном случае - пустой Optional, что означает, что можно проводить
     * само обновление.
     */
    @Override
    public final Optional<MassResult<R>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        validateModelChangesInternal();
        if (result != null) {
            return Optional.of(result);
        }
        onModelChangesValidated(createModelChangesValidatedStep());

        applyChanges();
        if (result != null) {
            return Optional.of(result);
        }
        onChangesApplied(createChangesAppliedStep());

        validateAppliedChangesInternal();
        if (result != null) {
            return Optional.of(result);
        }
        onAppliedChangesValidated(createAppliedChangesValidatedStep());

        return Optional.empty();
    }

    /**
     * Если результат готов, то возвращает его, в противном случае - пустой Optional.
     */
    @Override
    public final Optional<MassResult<R>> getResult() {
        return Optional.ofNullable(result);
    }

    protected boolean isPrepared() {
        return prepared;
    }

    /**
     * Непосредственно обновление.
     * <p>
     * Метод можно вызвать только после вызова метода {@link #prepare()},
     * который проводит валидацию и создает необходимые для обновления артефакты.
     * <p>
     * Метод можно вызвать только в том случае, если валидация прошла успешно
     * (вызов {@link #prepare} вернул пустой Optional.
     * <p>
     * Перед вызовом данного метода необходимо вызвать метод {@link #prepare()}.
     * <p>
     * Метод можно вызвать только один раз
     *
     * @return результат операции
     */
    @Override
    public final MassResult<R> apply() {
        checkApplyOrCancelPreconditions();
        Set<Integer> allIndexesSet = ContiguousSet.create(
                Range.closedOpen(0, modelChanges.size()), DiscreteDomain.integers());
        applyInternal(allIndexesSet);
        postProcessResult(result);
        return result;
    }

    @Override
    public MassResult<R> apply(Set<Integer> elementIndexesToApply) {
        checkApplyOrCancelPreconditions();
        checkArgument(Sets.difference(elementIndexesToApply, validAppliedChanges.keySet()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");
        applyInternal(elementIndexesToApply);
        return result;
    }

    @Override
    public final MassResult<R> cancel() {
        checkApplyOrCancelPreconditions();
        cancelInternal();
        return result;
    }

    /**
     * Получение индексов тех элементов, которые успешно провалидированы и готовы к применению операции.
     * <p>
     * Метод можно вызвать только после вызова метода {@link #prepare()}
     *
     * @return индексы валидных элементов
     */
    @Override
    public Set<Integer> getValidElementIndexes() {
        checkState(prepared, "prepare() must be called before getValidElementIndexes()");

        if (validAppliedChanges != null) {
            return validAppliedChanges.keySet();
        } else {
            return emptySet();
        }
    }

    protected boolean isExecuted() {
        return executed;
    }

    /**
     * Запомнить (обработать) изменение свойств моделей.
     *
     * @param modelProperty изменяемое свойство
     * @param indexValueMap соответствие индексов моделей и новых значений для их изменяемого свойства
     * @param <T>           тип значения свойства
     * @param <X>           тип модели, в которой объявлено свойство
     * @throws IllegalArgumentException Если модель хотя бы по одному из индексов не содержит указанное свойство
     *                                  и изменения не могут быть применены. В этом случае ни одна модель
     *                                  при вызове метода не изменится.
     */
    protected <T, X extends ModelWithId> void modifyProperty(ModelProperty<? super X, T> modelProperty,
                                                             Map<Integer, T> indexValueMap) {

        indexValueMap.keySet().forEach(i -> assertModificationIsAcceptable(i, modelProperty));

        for (Map.Entry<Integer, T> entry : indexValueMap.entrySet()) {
            int i = entry.getKey();
            T value = entry.getValue();
            ModelChanges<M> modelChangesItem = modelChanges.get(i);

            @SuppressWarnings("unchecked")
            ModelChanges<X> castedMc = (ModelChanges<X>) modelChangesItem;
            castedMc.process(value, modelProperty);

            @SuppressWarnings("unchecked")
            AppliedChanges<X> appliedChanges = (AppliedChanges<X>) appliedChangesForValidModelChanges.get(i);
            if (appliedChanges != null) {
                appliedChanges.modify(modelProperty, value);
            }
        }
    }

    private void assertModificationIsAcceptable(int changesIndex, ModelProperty<? extends Model, ?> property) {
        ModelChanges<M> modelChangesItem = modelChanges.get(changesIndex);
        checkArgument(isPropertyModificationAcceptable(property, modelChangesItem),
                "No such property in the model at specified index.\n"
                        + "index: %s, model type: %s, property name: %s",
                changesIndex, modelChangesItem.getModelType().getSimpleName(), property.name());
    }

    private boolean isPropertyModificationAcceptable(ModelProperty<? extends Model, ?> property,
                                                     ModelChanges<M> changes) {
        return property.getModelClass().isAssignableFrom(changes.getModelType());
    }

    private void applyModelChangesValidationResult(ValidationResult<List<ModelChanges<M>>, Defect> validationResult) {
        modelChangesValidationResult = cloneValidationResultSubNodesWithIssues(validationResult);
        validModelChanges = getValidItemsWithIndex(modelChangesValidationResult);

        if (modelChangesValidationResult.hasErrors() ||
                validModelChanges.isEmpty() ||
                (isFull(applicability) && modelChangesValidationResult.hasAnyErrors())) {
            List<R> results = Collections.nCopies(modelChanges.size(), null);
            ValidationResult<List<M>, Defect> vr =
                    modelChangesValidationToModelValidation(modelChangesValidationResult, emptyList(),
                            modelStubCreator);
            this.validationResult = vr;
            this.result = MassResult.brokenMassAction(results, vr);
        }
    }

    private void validateModelChangesInternal() {
        applyModelChangesValidationResult(validateModelChanges(modelChanges));
    }

    /**
     * Проводит валидацию списка ModelChanges (первый шаг валидации). Требуется реализовывать в потомках.
     *
     * @param modelChangesList - список ModelChanges, который необходимо провалидировать
     * @return результат валидации списка ModelChanges
     */
    protected abstract ValidationResult<List<ModelChanges<M>>, Defect> validateModelChanges(
            List<ModelChanges<M>> modelChangesList);

    /**
     * Метод можно реализовать в потомке, если требуется выполнить действия между первым
     * и вторым шагами валидации.
     * <p>
     * Метод вызывается только в случае, если после первого шага валидации результат операции еще не готов
     * (если первый шаг валидации не забраковал всю операцию).
     * <p>
     * Все данные, доступные методу на данном этапе, см. в интерфейсе
     * {@link ModelChangesValidatedStep}.
     *
     * @param modelChangesValidatedStep объект, через который метод может получить доступ
     *                                  к имеющимся на данный момент артефактам
     *                                  (например, к результату валидации списка ModelChanges).
     */
    protected void onModelChangesValidated(ModelChangesValidatedStep<M> modelChangesValidatedStep) {
    }


    /**
     * Провалидировать изменения перед непосредственным применением к модели
     *
     * @return null в том случае если можно использовать исходный результат валидации изменений,
     * в противном случае новый результат валидации
     */
    protected ValidationResult<List<ModelChanges<M>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<M>>, Defect> preValidateResult,
            Map<Long, M> models) {
        return null;
    }

    private void applyChanges() {
        Collection<Long> idsOfValidModelChanges = mapList(validModelChanges.values(), ModelChanges::getId);
        Map<Long, M> models = StreamEx.of(getModels(idsOfValidModelChanges))
                .mapToEntry(ModelWithId::getId, Function.identity())
                .toMap();
        checkModelsCollectionsMatch(models.values(), validModelChanges.values());

        ValidationResult<List<ModelChanges<M>>, Defect> validationResult =
                validateModelChangesBeforeApply(modelChangesValidationResult, models);
        if (validationResult != null) {
            applyModelChangesValidationResult(validationResult);
        }

        if (result == null) {
            appliedChangesForValidModelChanges = getAppliedChangesForValidModelChanges(validModelChanges, models);
        }
    }

    protected Map<Integer, AppliedChanges<M>> getAppliedChangesForValidModelChanges(
            Map<Integer, ModelChanges<M>> validModelChanges,
            Map<Long, M> models) {
        return EntryStream.of(validModelChanges)
                .mapValues(oneModelChanges -> {
                    M model = models.get(oneModelChanges.getId());
                    var applicableSensitiveProperties = getApplicableSensitiveProperties(model);
                    return oneModelChanges.applyTo(model, applicableSensitiveProperties, customModelEquals);
                })
                .toMap();
    }

    private Set<ModelProperty<? super M, ?>> getApplicableSensitiveProperties(M model) {
        //noinspection unchecked
        return sensitiveProperties.stream()
                .filter(property -> property.getModelClass().isAssignableFrom(model.getClass()))
                .map(x -> (ModelProperty<? super M, ?>) x)
                .collect(Collectors.toSet());
    }

    /**
     * Получает модели для тех ModelChanges, для которых успешно прошел первый этап валидации,
     * для последующего применения к ним изменений и получения объектов {@link AppliedChanges}.
     * Требуется реализовывать в потомках.
     *
     * @param ids коллекция id моделей, которые необходимо загрузить
     * @return коллекция моделей
     */
    protected abstract Collection<M> getModels(Collection<Long> ids);

    public ValidationResult<List<M>, Defect> getValidationResult() {
        return validationResult;
    }

    /**
     * Метод можно реализовать в потомке, если требуется выполнить некоторые действия сразу после того,
     * как модели загружены и к ним применены изменения. На этом шаге модели с примененными изменениями
     * еще не провалидированы, и методу доступен список AppliedChanges для всех успешно провалидированных
     * объектов ModelChanges с помощью метода {@link ChangesAppliedStep#getAppliedChangesForValidModelChanges()}.
     * Все данные, доступные методу на данном этапе, см. в интерфейсе {@link ChangesAppliedStep}.
     *
     * @param changesAppliedStep объект, через который метод может получить доступ
     *                           к имеющимся на данный момент артефактам
     *                           (например, к списку AppliedChanges для успешно провалидированных ModelChanges).
     */
    @SuppressWarnings("unused")
    protected void onChangesApplied(ChangesAppliedStep<M> changesAppliedStep) {
    }

    private void validateAppliedChangesInternal() {
        validationResult = convertModelChangesValidation();
        validationResult = validateAppliedChanges(validationResult, appliedChangesForValidModelChanges);

        checkState(!validationResult.hasErrors(),
                "applied changes validation stage must not generate operation-level errors");

        validAppliedChanges = extractOnlyValidAppliedChangesWithIndex(
                appliedChangesForValidModelChanges, validationResult);

        if (validAppliedChanges.isEmpty() ||
                (isFull(applicability) && validationResult.hasAnyErrors())) {
            List<R> results = Collections.nCopies(modelChanges.size(), null);
            this.result = MassResult.brokenMassAction(results, validationResult);
        }
    }

    private ValidationResult<List<M>, Defect> convertModelChangesValidation() {
        Collection<M> models = mapList(appliedChangesForValidModelChanges.values(), AppliedChanges::getModel);
        return modelChangesValidationToModelValidation(modelChangesValidationResult, models, modelStubCreator);
    }

    /**
     * Метод валидирует модели с примененными к ним изменениями.
     * Работает также как {@link #validateAppliedChanges(ValidationResult)},
     * но дополнительно передаёт applied changes для валидных моделей (с ключом по индексу модели).
     */
    protected ValidationResult<List<M>, Defect> validateAppliedChanges(
            ValidationResult<List<M>, Defect> validationResult,
            Map<Integer, AppliedChanges<M>> appliedChangesForValidModelChanges) {
        return validateAppliedChanges(validationResult);
    }


    /**
     * Метод валидирует модели с примененными к ним изменениями. В качестве входных моделей
     * могут быть как полноценные модели с примененными к ним изменениями для успешно
     * провалидированных ModelChanges, так и пустые заглушки с выставленными id для
     * ModelChanges с ошибками. Поэтому, при валидации следует учитывать это и валидировать
     * только те модели, у которых нет ошибок (используя {@link When#isValid()}.
     *
     * @param validationResult результат валидации списка моделей, который необходимо взять
     *                         за основу и проводить на нем валидацию. Он получается
     *                         путем преобразования из результата валидации списка ModelChanges.
     *                         Соответственно, он содержит ошибки в элементах, соответствующим
     *                         ошибочным ModelChanges по индексам.
     * @return результат валидации списка моделей.
     */
    protected ValidationResult<List<M>, Defect> validateAppliedChanges(
            ValidationResult<List<M>, Defect> validationResult) {
        return validationResult;
    }

    /**
     * Метод можно реализовать в потомке для выполнения дополнительных действий
     * после валидации моделей с примененными изменениями.
     * <p>
     * Метод вызывается только в случае, если после второго шага валидации результат операции еще не готов
     * (если второй шаг валидации не забраковал всю операцию).
     * <p>
     * Все данные, доступные методу на данном этапе см. в интерфейсе {@link AppliedChangesValidatedStep}.
     *
     * @param appliedChangesValidatedStep объект, через который метод может получить доступ
     *                                    к имеющимся на данный момент артефактам
     *                                    (например, к результату валидации моделей).
     */
    @SuppressWarnings("unused")
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<M> appliedChangesValidatedStep) {
    }

    private void applyInternal(Set<Integer> elementIndexesToApply) {
        executed = true;
        applicableAppliedChanges = EntryStream.of(validAppliedChanges)
                .filterKeys(elementIndexesToApply::contains)
                .toMap();
        beforeExecution(createExecutionStep());
        Map<Integer, R> executeResult = execute(createExecutionStep());
        checkState(executeResult.keySet().equals(applicableAppliedChanges.keySet()),
                "execute() returned map with invalid keySet (should be equals)");
        Set<Integer> canceledElements = Sets.difference(validAppliedChanges.keySet(), elementIndexesToApply);
        List<R> results = new ArrayList<>();
        for (int i = 0; i < modelChanges.size(); ++i) {
            results.add(executeResult.get(i));
        }
        result = MassResult.successfulMassAction(results, validationResult, canceledElements);
        afterExecution(createExecutionStep());
    }

    private void cancelInternal() {
        executed = true;
        List<R> results = Collections.nCopies(modelChanges.size(), null);
        // все валидные элементы помечаем отменёнными, т.к. для них не выполнялась операция
        result = MassResult.successfulMassAction(results, validationResult, validAppliedChanges.keySet());
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
    }

    /**
     * Выполняется непосредственно перед сохранением валидных изменений. Метод можно реализовать в потомке для
     * выполнения дополнительных действий с {@code validAppliedChanges}. Список изменений, передающийся в этот
     * метод точно такой же с которым выполняется {@link #execute(ExecutionStep)}
     *
     * @param executionStep объект, через который метод может получить доступ
     *                      к имеющимся на данный момент артефактам
     *                      (например, к списку AppliedChanges для выполнения).
     */
    @SuppressWarnings("unused")
    protected void beforeExecution(ExecutionStep<M> executionStep) {
    }

    /**
     * Абстрактный метод сохранения валидных изменений. Входящий и исходящий маппинги должны иметь одинаковый keySet
     *
     * @param executionStep объект, через который метод может получить доступ
     *                      к имеющимся на данный момент артефактам
     *                      (например, к списку AppliedChanges для выполнения).
     * @return маппинг тех же самых индексов на результат
     */
    protected abstract Map<Integer, R> execute(ExecutionStep<M> executionStep);

    /**
     * Выполняется непосредственно после сохранения изменений в методе {@link #execute(ExecutionStep)}. Метод можно
     * реализовать в потомке для выполнения дополнительных действий после сохранения.
     */
    @SuppressWarnings("unused")
    protected void afterExecution(ExecutionStep<M> executionStep) {
    }

    /**
     * Дополнительный этап обработки результата, если нужно изменить значение после выполнения операции.
     * Метод можно реализовать в потомке для выполнения дополнительных действий после сохранения.
     */
    protected void postProcessResult(MassResult<R> massResult) {
    }

    private ModelChangesValidatedStep<M> createModelChangesValidatedStep() {
        return new ModelChangesValidatedStepInner();
    }

    private ChangesAppliedStep<M> createChangesAppliedStep() {
        return new ChangesAppliedStepInner();
    }

    private AppliedChangesValidatedStep<M> createAppliedChangesValidatedStep() {
        return new AppliedChangesValidatedStepInner();
    }

    private ExecutionStep<M> createExecutionStep() {
        return new ExecutionStepInner();
    }

    private class ModelChangesValidatedStepInner implements ModelChangesValidatedStep<M> {
        @Override
        public List<ModelChanges<M>> getModelChanges() {
            return modelChanges;
        }

        @Override
        public ValidationResult<List<ModelChanges<M>>, Defect> getModelChangesValidationResult() {
            return modelChangesValidationResult;
        }

        @Override
        public void setModelChangesValidationResult(
                ValidationResult<List<ModelChanges<M>>, Defect> modelChangesValidationResult) {
            AbstractUpdateOperation.this.modelChangesValidationResult = modelChangesValidationResult;
        }

        @Override
        public Collection<ModelChanges<M>> getValidModelChanges() {
            return validModelChanges.values();
        }

        @Override
        public Map<Integer, ModelChanges<M>> getValidModelChangesWithIndex() {
            return validModelChanges;
        }
    }

    private class ChangesAppliedStepInner extends ModelChangesValidatedStepInner implements ChangesAppliedStep<M> {
        @Override
        public Collection<AppliedChanges<M>> getAppliedChangesForValidModelChanges() {
            return appliedChangesForValidModelChanges.values();
        }

        @Override
        public Map<Integer, AppliedChanges<M>> getAppliedChangesForValidModelChangesWithIndex() {
            return appliedChangesForValidModelChanges;
        }
    }

    private class AppliedChangesValidatedStepInner extends ChangesAppliedStepInner implements AppliedChangesValidatedStep<M> {
        @Override
        public ValidationResult<List<M>, Defect> getValidationResult() {
            return validationResult;
        }

        @Override
        public Collection<AppliedChanges<M>> getValidAppliedChanges() {
            return validAppliedChanges.values();
        }

        @Override
        public Map<Integer, AppliedChanges<M>> getValidAppliedChangesWithIndex() {
            return validAppliedChanges;
        }
    }

    private class ExecutionStepInner extends AppliedChangesValidatedStepInner implements ExecutionStep<M> {
        @Override
        public Collection<AppliedChanges<M>> getAppliedChangesForExecution() {
            return applicableAppliedChanges.values();
        }

        @Override
        public Map<Integer, AppliedChanges<M>> getAppliedChangesForExecutionWithIndex() {
            return applicableAppliedChanges;
        }
    }

    protected Map<Integer, ModelChanges<M>> getValidModelChanges() {
        return validModelChanges;
    }
}
