package ru.yandex.direct.organizations.swagger;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish;
import ru.yandex.direct.organizations.swagger.model.CompanyChain;
import ru.yandex.direct.organizations.swagger.model.CompanyEmail;
import ru.yandex.direct.organizations.swagger.model.CompanyName;
import ru.yandex.direct.organizations.swagger.model.CompanyPhone;
import ru.yandex.direct.organizations.swagger.model.CompanyRubric;
import ru.yandex.direct.organizations.swagger.model.CompanyUrl;
import ru.yandex.direct.organizations.swagger.model.LocalizedString;
import ru.yandex.direct.organizations.swagger.model.MetrikaData;
import ru.yandex.direct.organizations.swagger.model.Photo;
import ru.yandex.direct.organizations.swagger.model.PubApiCompaniesData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompany;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanyData;
import ru.yandex.direct.organizations.swagger.model.PublishingStatus;
import ru.yandex.direct.organizations.swagger.model.TycoonRubricDefinition;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.JsonUtils.MAPPER;

@ParametersAreNonnullByDefault
public class OrganizationInfoConverters {

    private static final String EMPTY_COMPANY_NAME = "---";

    private static final Logger logger = LoggerFactory.getLogger(OrganizationInfoConverters.class);

    /**
     * Добавляет no_similar=true к ссылке
     */
    public static String getProfileUrl(String profileUrlFromSprav) {
        if (profileUrlFromSprav == null || profileUrlFromSprav.isEmpty()) {
            return profileUrlFromSprav;
        }
        try {
            URIBuilder uriBuilder = new URIBuilder(profileUrlFromSprav);
            uriBuilder.addParameter("no_similar", "true");
            return uriBuilder.toString();
        } catch (Exception e) {
            return profileUrlFromSprav;
        }
    }

    /**
     * Конвертирует статус публикации организации в ее статус модерации в ядре.
     */
    public static OrganizationStatusPublish toCoreStatusPublish(PublishingStatus apiStatus) {
        switch (apiStatus) {
            case PUBLISH:
                return OrganizationStatusPublish.PUBLISHED;
            case MOVED:
                return OrganizationStatusPublish.MOVED;
            case TEMPORARILY_CLOSED:
                return OrganizationStatusPublish.TEMPORARILY_CLOSED;
            // Пока API Справочника возвращает все (почти) невалидные статусы как CLOSED,
            //   придется CLOSED считать как UNPUBLISHED.
            default:
                return OrganizationStatusPublish.UNPUBLISHED;
        }
    }

    /**
     * Выбор имени компании. Если название компании не удалось вычислить, то возвращается заглушка
     */
    @Nonnull
    static String getCompanyName(@Nullable Collection<CompanyName> names, String language) {
        String companyName = getCompanyNameInternal(names, language);
        return companyName == null ? EMPTY_COMPANY_NAME : companyName;
    }

    /**
     * Выбор имени компании.
     */
    @Nullable
    private static String getCompanyNameInternal(@Nullable Collection<CompanyName> names, String language) {
        if (isEmpty(names)) {
            return null;
        }
        Map<String, String> namesByLanguage = StreamEx.of(names)
                .filter(name -> name.getType() == CompanyName.TypeEnum.MAIN)
                .map(CompanyName::getValue)
                .flatMap(StreamEx::of)
                .toMap(LocalizedString::getLocale, LocalizedString::getValue);
        return namesByLanguage.getOrDefault(language, namesByLanguage.values().iterator().next());
    }

    /**
     * Выбор email'а компании.
     */
    @Nullable
    static String getCompanyEmail(@Nullable Collection<CompanyEmail> emails) {
        if (emails == null) {
            return null;
        }
        return StreamEx.of(emails).findFirst().map(CompanyEmail::getValue).orElse(null);
    }

    /**
     * Выбор preview ссылки компании.
     * На тестинге отсекаем продовым оргам mdst на mds, чтобы получить рабочее продовое превью
     * Определение продовости сделано костылём по содержанию в ссылке на профиль l7test
     */
    @Nullable
    static String getPreviewHref(@Nullable Photo logo, String businessProfilelink) {
        if (logo == null) {
            return null;
        }
        boolean isProd = businessProfilelink == null || !businessProfilelink.contains("l7test");
        String previewPrepared = String.format(logo.getUrlTemplate(), "S");
        if (isProd) {
            previewPrepared = previewPrepared.replace("mdst", "mds");
        }
        return previewPrepared;
    }

    private static List<CompanyPhone> getCompanyPhones(@Nullable List<CompanyPhone> phones) {
        return phones == null ? emptyList() : phones;
    }

    @Nullable
    static Long getCompanyChainId(@Nullable CompanyChain chain) {
        return Optional.ofNullable(chain).map(CompanyChain::getId).orElse(null);
    }

    @Nullable
    static String getRubric(PubApiCompanyData company, String language) {
        if (isEmpty(company.getRubricDefs())) {
            return null;
        }
        Map<String, String> namesByLanguage = StreamEx.of(company.getRubricDefs().values())
                .map(TycoonRubricDefinition::getNames)
                .flatMap(StreamEx::of)
                .toMap(LocalizedString::getLocale, LocalizedString::getValue, (s1, s2) -> s1 + ", " + s2);
        return namesByLanguage.getOrDefault(language, namesByLanguage.values().iterator().next());
    }

    @Nullable
    static String getRubric(PubApiCompany company, Map<Long, String> localizedRubricNames) {
        return StreamEx.of(company.getRubrics())
                .map(CompanyRubric::getRubricId)
                .map(localizedRubricNames::get)
                .findAny(Objects::nonNull)
                .orElse(null);
    }

    @Nonnull
    static Map<Long, String> getLocalizedRubricNames(Map<String, TycoonRubricDefinition> defs, String language) {
        if (isEmpty(defs)) {
            return emptyMap();
        }
        return StreamEx.of(defs.values())
                .toMap(
                        TycoonRubricDefinition::getId,
                        org -> {
                            List<LocalizedString> names = org.getNames();
                            if (isEmpty(names)) {
                                return null;
                            }
                            return StreamEx.of(names)
                                    .findAny(n -> n.getLocale().equals(language))
                                    .orElse(names.iterator().next())
                                    .getValue();
                        }
                );
    }

    @Nonnull
    static List<String> convertCompanyUrls(@Nullable Collection<CompanyUrl> urls) {
        if (urls == null) {
            return emptyList();
        }
        return StreamEx.of(urls)
                .filter(url -> url.isHide() == null || !url.isHide())
                .map(CompanyUrl::getValue)
                .nonNull()
                .toList();
    }

    public static List<OrganizationInfo> convertMultipleOrganizationsInfo(PubApiCompaniesData companies,
                                                                          String language) {
        return convertMultipleOrganizationsInfo(OrganizationInfo::new, companies, language);
    }

    /**
     * Преобразование организаций из ответа справочника во внутреннее представление.
     *
     * @param language Язык, на котором отображается название организации.
     * @return Список данных об организациях.
     */
    @Nonnull
    public static <T extends OrganizationInfo> List<T> convertMultipleOrganizationsInfo(
            Supplier<T> newInstanceSupplier, PubApiCompaniesData companies, String language) {
        ArrayList<T> result = new ArrayList<>();
        if (companies == null) {
            return result;
        }
        Map<Long, String> localizedRubricNames = getLocalizedRubricNames(companies.getRubricDefs(), language);
        for (PubApiCompany company : companies.getCompanies()) {
            if (company == null) {
                continue;
            }
            T info = newInstanceSupplier.get();
            List<CompanyPhone> companyPhones = getCompanyPhones(company.getPhones());
            info
                    .withAddress(company.getAddress())
                    .withCompanyName(getCompanyName(company.getNames(), language))
                    .withPhone(companyPhones.isEmpty() ? null : companyPhones.get(0))
                    .withPhones(companyPhones)
                    .withCabinetUrl(company.getLkLink())
                    .withProfileUrl(getProfileUrl(company.getBusinessProfileLink()))
                    .withEmail(getCompanyEmail(company.getEmails()))
                    .withWorkIntervals(Optional.ofNullable(company.getWorkIntervals()).orElse(emptyList()))
                    .withRubric(getRubric(company, localizedRubricNames))
                    .withUrls(convertCompanyUrls(company.getUrls()))
                    .withIsOnline(company.isIsOnline())
                    .withMetrikaData(company.getMetrikaData())
                    .withPreviewHref(getPreviewHref(company.getLogo(), company.getBusinessProfileLink()))
                    .withPermalinkId(company.getId())
                    .withStatusPublish(Optional.ofNullable(company.getPublishingStatus())
                            .map(OrganizationInfoConverters::toCoreStatusPublish)
                            .orElse(OrganizationStatusPublish.UNKNOWN))
                    .withChainId(getCompanyChainId(company.getChain()));
            result.add(info);
        }
        return result;
    }

    @Nonnull
    public static OrganizationInfo convertSingleOrganizationInfo(Long permalinkId, String language,
                                                                 PubApiCompanyData company) {
        var info = new OrganizationInfo();
        String companyName = getCompanyName(company.getNames(), language);
        String email = getCompanyEmail(company.getEmails());
        List<CompanyPhone> companyPhones = getCompanyPhones(company.getPhones());
        info
                .withAddress(company.getAddress())
                .withCompanyName(companyName)
                .withPhone(companyPhones.isEmpty() ? null : companyPhones.get(0))
                .withPhones(companyPhones)
                .withCabinetUrl(company.getLkLink())
                .withProfileUrl(getProfileUrl(company.getBusinessProfileLink()))
                .withWorkIntervals(Optional.ofNullable(company.getWorkIntervals()).orElse(emptyList()))
                .withEmail(email)
                .withRubric(getRubric(company, language))
                .withUrls(convertCompanyUrls(company.getUrls()))
                .withIsOnline(company.isIsOnline())
                .withMetrikaData(company.getMetrikaData())
                .withPreviewHref(getPreviewHref(company.getLogo(), company.getBusinessProfileLink()))
                .withPermalinkId(permalinkId)
                .withStatusPublish(Optional.ofNullable(company.getPublishingStatus())
                        .map(OrganizationInfoConverters::toCoreStatusPublish)
                        .orElse(OrganizationStatusPublish.UNKNOWN))
                .withChainId(getCompanyChainId(company.getChain()));
        return info;
    }

    /**
     * Позволяет по ответу Справочника и списку запрашиваемых пермалинков восстановить данные организаций-дубликатов
     * при наличии данных об их головах
     *
     * @param companiesData - ответ API Справочника
     * @param permalinkIds  - запрашиваемые пермалинки
     * @return дополненный ответ
     */
    @Nonnull
    public static PubApiCompaniesData resolveDuplicates(PubApiCompaniesData companiesData, Set<Long> permalinkIds) {
        // Резолвить что-либо можно только если в запросе было несколько пермалинков
        //   и в ответе есть данные хотя бы по одной организаци.
        //   Иначе можно выходить из функции.
        List<PubApiCompany> companies = filterList(companiesData.getCompanies(), Objects::nonNull);
        if (permalinkIds.size() <= 1 || companies == null || companies.isEmpty()) {
            return companiesData;
        }
        Set<Long> foundPermalinks = listToSet(companies, PubApiCompany::getId); // найденные пермалинки
        if (foundPermalinks.size() == permalinkIds.size()) {
            return companiesData; // Если нашли все пермалинки, то выходим.
        }
        logger.info("Try to resolve duplicates with permalinks " + permalinkIds);

        // Собираем мапу пермалинк -> данные оригинала
        Map<Long, PubApiCompany> duplicateToHead = StreamEx.of(companies)
                .filter(company -> company.getDuplicates() != null)
                .flatMapToEntry(head -> StreamEx.of(head.getDuplicates())
                        .mapToEntry(
                                duplicate_id -> head
                        ).toMap()
                )
                // Если один пермалинк будет дубликатом сразу у двух оригиналов (что не должно быть),
                //   то берем произвольный.
                .distinctKeys()
                .toMap();

        StreamEx.of(permalinkIds)
                .filter(id -> !foundPermalinks.contains(id)) // найденные пермалинки пропускаем
                .mapToEntry(
                        duplicateToHead::get
                ) // для каждого ненайденного пермалинка ищем данные оригинала
                .nonNullValues() // оставляем только дубли, для которых есть информация по оригиналу
                .mapKeyValue(
                        (id, company) -> copyCompanyData(company)
                                .id(id)
                                .publishingStatus(PublishingStatus.DUPLICATE)) // копируем оригинал, подменя id и статус
                .forEach(duplicate -> {
                    logger.info("Resolved duplicate: " + duplicate);
                    companiesData.addCompaniesItem(duplicate); // добавляем данные о дубле в ответ
                });

        return companiesData;
    }

    /**
     * Копируем данные по организации в новый объект.
     *
     * @param company - копируемая кампания
     * @return копия
     */
    public static PubApiCompany copyCompanyData(PubApiCompany company) {
        try {
            byte[] data = MAPPER.writeValueAsBytes(company);
            return MAPPER.readValue(data, PubApiCompany.class);
        } catch (Exception e) {
            logger.error("There is broken coping of company " + company, e);
            throw new RuntimeException(e);
        }
    }

    @Nonnull
    public static PubApiCompaniesData updateLink(PubApiCompaniesData companies) {
        for (PubApiCompany company : companies.getCompanies()) {
            company.setLkLink(updateLink(company.getLkLink()));
        }
        return companies;
    }

    @Nonnull
    public static PubApiCompanyData updateLink(PubApiCompanyData company) {
        company.setLkLink(updateLink(company.getLkLink()));
        return company;
    }

    @Nonnull
    public static PubApiCompaniesData updatePublishingStatus(PubApiCompaniesData companies) {
        for (PubApiCompany company : companies.getCompanies()) {
            if (company.getId() != null
                    && company.getHeadPermalink() != null
                    && !company.getId().equals(company.getHeadPermalink())) {
                company.setPublishingStatus(PublishingStatus.DUPLICATE);
            }
        }
        return companies;
    }

    @Nonnull
    public static PubApiCompanyData updatePublishingStatus(PubApiCompanyData company) {
        if (company.getId() != null
                && company.getHeadPermalink() != null
                && !company.getId().equals(company.getHeadPermalink())) {
            company.setPublishingStatus(PublishingStatus.DUPLICATE);
        }
        return company;
    }

    @Nullable
    public static Long getMetrikaCounterId(@Nullable MetrikaData metrikaData) {
        if (metrikaData == null) {
            return null;
        }
        String counter = metrikaData.getCounter();
        if (StringUtils.isEmpty(counter)) {
            return null;
        }
        try {
            return Long.valueOf(counter);
        } catch (NumberFormatException e) {
            logger.warn("Cannot parse metrika counter '{}' to number", counter);
            return null;
        }
    }

    private static String updateLink(String link) {
        if (StringUtils.isEmpty(link)) {
            return link;
        }
        try {
            URI uri = new URI(link);
            String query = uri.getQuery();
            if (query == null) {
                query = "source=direct";
            } else {
                query += "&source=direct";
            }
            return new URI(uri.getScheme(), uri.getAuthority(), uri.getPath(), query, uri.getFragment()).toString();
        } catch (URISyntaxException ex) {
            return link;
        }
    }
}
