package ru.yandex.direct.organizations.swagger;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.altay.model.language.LanguageOuterClass.Language;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.annotations.ResponseHandler;
import ru.yandex.direct.http.smart.converter.JsonResponseConverter;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.CallsPack;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.GET;
import ru.yandex.direct.http.smart.http.HeaderMap;
import ru.yandex.direct.http.smart.http.PATCH;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.PartibleQuery;
import ru.yandex.direct.http.smart.http.Path;
import ru.yandex.direct.http.smart.http.Query;
import ru.yandex.direct.organizations.swagger.model.MetrikaData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompaniesData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompany;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanyData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanySearchBody;
import ru.yandex.direct.organizations.swagger.model.PubApiCreateCompanyRequest;
import ru.yandex.direct.organizations.swagger.model.PubApiUserAccessLevel;
import ru.yandex.direct.organizations.swagger.model.SpravResponse;
import ru.yandex.direct.organizations.swagger.model.UpdateOrganizationRequest;
import ru.yandex.direct.organizations.swagger.model.UpdateOrganizationResponse;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.io.http.HttpHeaderNames;

@ParametersAreNonnullByDefault
public interface OrganizationsApi {

    /**
     * Получение данных по одной компании.
     *
     * @param permalink ID организации в справочнике
     * @param expand    Какие поля нужно вернуть. Формат - {@code expand=a&expand=b&expand=...}
     * @param headers   Хедеры запроса, обычно содержат:<ul>
     *                  <li>язык из {@link Language} в хедере {@link HttpHeaderNames#ACCEPT_LANGUAGE}</li>
     *                  <li>пользовательский TVM-тикет в хедере {@link TvmHeaders#USER_TICKET}</li>
     *                  </ul>
     */
    @GET("/v1/companies/{permalink}")
    @ResponseHandler(parserClass = PubApiCompanyDataResponseConverter.class)
    Call<PubApiCompanyData> getSingleCompanyByPermalink(
            @Path("permalink") Long permalink,
            @Query("expand") Collection<String> expand,
            @HeaderMap Map<String, String> headers);

    /**
     * Получение данных по нескольким компаниям с фильтрацией по uid.
     *
     * @param permalinks      Список пермалинков организаций
     * @param uids            Список uid-ов главных представителей организаций
     * @param expand          Какие поля нужно вернуть. Формат - {@code expand=a&expand=b&expand=...}
     * @param userAccessLevel Уровень доступа к информации по организациям.
     *                        Должно быть равно "any", т.к. значение "owned" не сочетается с фильтрацией по permalinks.
     * @param headers         Хедеры запроса, обычно содержат:<ul>
     *                        <li>язык из {@link Language} в хедере {@link HttpHeaderNames#ACCEPT_LANGUAGE}</li>
     *                        <li>пользовательский TVM-тикет в хедере {@link TvmHeaders#USER_TICKET}</li>
     *                        </ul>
     */
    @GET("/v1/companies")
    @ResponseHandler(parserClass = PubApiCompaniesDataResponseConverter.class)
    CallsPack<PubApiCompaniesData> getCompaniesByPermalinkAndUid(
            @PartibleQuery("filter:permalink") Collection<Long> permalinks,
            @Query("filter:uid") Collection<Long> uids,
            @Query("expand") Collection<String> expand,
            @Query("filter:user_access_level") PubApiUserAccessLevel userAccessLevel,
            @HeaderMap Map<String, String> headers);

    /**
     * Резервирование permalinkId. Рекомендуется использовать перед созданием организации.
     * Возвращает список, так как ручка умеет резервировать сразу пачку ID
     */
    @POST("/v1/permalinks")
    @ResponseHandler(parserClass = JsonResponseConverter.class, expectedCodes = {201})
    @Json
    Call<List<Long>> reservePermalinks();

    /**
     * Создание организации
     */
    @POST("/v1/companies")
    @ResponseHandler(parserClass = PubApiCompanyResponseConverter.class, expectedCodes = {201, 422})
    Call<SpravResponse<PubApiCompany>> createCompany(
            @Body @Json PubApiCreateCompanyRequest createBody,
            @HeaderMap Map<String, String> headers);

    /**
     * Получение данных по нескольким компаниям с фильтрацией по нескольким параметрам.
     */
    @POST("/v1/companies/search")
    @ResponseHandler(parserClass = PubApiCompaniesDataResponseConverter.class)
    Call<PubApiCompaniesData> getCompaniesByPermalinkAndUidAsPost(
            @Body @Json PubApiCompanySearchBody searchBody,
            @HeaderMap Map<String, String> headers);

    /**
     * Получение данных о счетчиках метрики пермалинков по их id
     */
    @POST("/v1/metrika")
    @Json
    Call<List<MetrikaData>> getOrganizationsCountersData(@Body Collection<Long> permalinkIds,
                                                         @HeaderMap Map<String, String> headers);


    /**
     * Создание счётчика и возвращение данных о нём.
     * Если счетчик уже существует, то ручка вернет существующий.
     */
    @POST("/v3/companies/{permalink}/create-metrika")
    @Json
    Call<MetrikaData> createCounterByPermalink(@Path("permalink") Long permalink);

    /**
     * Обновление организации
     */
    @PATCH("/v3/companies/{permalink}")
    @ResponseHandler(parserClass = UpdateOrganizationResponseConverter.class, expectedCodes = {200, 204, 400, 404, 422})
    Call<UpdateOrganizationResponse> updateCompany(
            @Path("permalink") Long permalink,
            @Body @Json UpdateOrganizationRequest body,
            @HeaderMap Map<String, String> headers);

}
