package ru.yandex.direct.organizations.swagger;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.asynchttpclient.Param;
import org.asynchttpclient.Request;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.altay.model.language.LanguageOuterClass.Language;
import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.organizations.swagger.model.CreateOnlineOrgRequest;
import ru.yandex.direct.organizations.swagger.model.MetrikaData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompaniesData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompany;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanyData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanySearchBody;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanySearchFilter;
import ru.yandex.direct.organizations.swagger.model.PubApiCreateCompanyRequest;
import ru.yandex.direct.organizations.swagger.model.PubApiExtraCompanyField;
import ru.yandex.direct.organizations.swagger.model.PubApiUserAccessLevel;
import ru.yandex.direct.organizations.swagger.model.PublishingStatus;
import ru.yandex.direct.organizations.swagger.model.SpravResponse;
import ru.yandex.direct.organizations.swagger.model.UpdateOrganizationRequest;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.io.http.HttpHeaderNames;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static org.asynchttpclient.util.MiscUtils.isNonEmpty;
import static ru.yandex.direct.organizations.swagger.OrganizationInfoConverters.resolveDuplicates;
import static ru.yandex.direct.organizations.swagger.OrganizationInfoConverters.updateLink;
import static ru.yandex.direct.organizations.swagger.OrganizationInfoConverters.updatePublishingStatus;
import static ru.yandex.direct.organizations.swagger.model.PubApiUserAccessLevel.ANY;
import static ru.yandex.direct.organizations.swagger.model.PubApiUserAccessLevel.OWNED;
import static ru.yandex.direct.organizations.util.OrganizationUtils.mergePubApiCompaniesData;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Клиент к API справочника, получает данные об организациях.
 */
@ParametersAreNonnullByDefault
public class OrganizationsClient {
    private static final Set<Integer> KNOWN_ERROR_CODES = Set.of(
            HttpStatus.SC_INTERNAL_SERVER_ERROR,
            HttpStatus.SC_UNAUTHORIZED,
            HttpStatus.SC_UNPROCESSABLE_ENTITY
    );

    private static final List<String> TVM_TICKED_HEADERS = List.of(TvmHeaders.SERVICE_TICKET, TvmHeaders.USER_TICKET);

    private static final List<PubApiExtraCompanyField> DEFAULT_FIELDS_AS_ENUMS = List.of(
            PubApiExtraCompanyField.NAMES,
            PubApiExtraCompanyField.EMAILS,
            PubApiExtraCompanyField.ADDRESS,
            PubApiExtraCompanyField.PHONES,
            PubApiExtraCompanyField.CHAIN,
            PubApiExtraCompanyField.WORK_INTERVALS,
            PubApiExtraCompanyField.BUSINESS_PROFILE_LINK,
            PubApiExtraCompanyField.LK_LINK,
            PubApiExtraCompanyField.PUBLISHING_STATUS,
            PubApiExtraCompanyField.URLS,
            PubApiExtraCompanyField.IS_ONLINE,
            PubApiExtraCompanyField.RUBRICS,
            PubApiExtraCompanyField.RUBRIC_DEFS,
            PubApiExtraCompanyField.METRIKA_DATA);

    //нужно в getMultipleOrganizationsInfoByText для фильтрации. При отрыве добавить поле туда
    static {
        checkState(DEFAULT_FIELDS_AS_ENUMS.contains(PubApiExtraCompanyField.PUBLISHING_STATUS));
    }

    private static final List<String> DEFAULT_FIELDS = mapList(DEFAULT_FIELDS_AS_ENUMS,
            PubApiExtraCompanyField::getValue);

    private static final String GRANTS_FIELD = "grants";

    private static final Logger logger = LoggerFactory.getLogger(OrganizationsClient.class);

    private static final Map<String, Language> languagesByName = StreamEx.of(Language.values())
            .mapToEntry(Language::name, identity())
            .mapKeys(String::toLowerCase)
            .toMap();

    private final OrganizationsApi organizationsApi;
    private final int permalinkChunkSize;
    private final FetcherSettings searchByTextFetcherSettings;
    private final FetcherSettings writeRequestSettings;
    private final PpcProperty<Boolean> createMetrikaCounterProperty;

    public OrganizationsClient(String url,
                               ParallelFetcherFactory fetcherFactory,
                               TvmIntegration tvmIntegration,
                               FetcherSettings searchByTextFetcherSettings,
                               FetcherSettings writeRequestSettings,
                               PpcPropertiesSupport ppcPropertiesSupport,
                               boolean isProd,
                               int permalinkChunkSize) {
        this.permalinkChunkSize = permalinkChunkSize;
        this.searchByTextFetcherSettings = searchByTextFetcherSettings;
        this.writeRequestSettings = writeRequestSettings;
        organizationsApi = Smart.builder()
                .withParallelFetcherFactory(fetcherFactory)
                .useTvm(tvmIntegration,
                        isProd ? TvmService.SPRAV_API_PROD
                                : TvmService.SPRAV_API_TEST)
                .withProfileName("organizations_client")
                .withBaseUrl(url)
                .addHeaderConfigurator(headers -> headers.add("Content-type", "application/json"))
                .withResponseConverterFactory(
                        ResponseConverterFactory.builder()
                                .addConverters(
                                        new PubApiCompanyDataResponseConverter(),
                                        new PubApiCompaniesDataResponseConverter(),
                                        new PubApiCompanyResponseConverter(),
                                        new UpdateOrganizationResponseConverter()
                                )
                                .build()
                )
                .build()
                .create(OrganizationsApi.class);
        this.createMetrikaCounterProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.CREATE_ORGANIZATION_METRIKA_COUNTER, Duration.ofMinutes(1));

    }

    protected OrganizationsClient(
            OrganizationsApi organizationsApi,
            FetcherSettings searchByTextFetcherSettings,
            PpcProperty<Boolean> createMetrikaCounterProperty,
            int permalinkChunkSize
    ) {
        this.permalinkChunkSize = permalinkChunkSize;
        this.searchByTextFetcherSettings = searchByTextFetcherSettings;
        this.organizationsApi = organizationsApi;
        this.createMetrikaCounterProperty = createMetrikaCounterProperty;
        this.writeRequestSettings = null; // не переопределяем настройки
    }

    public static Optional<Language> getLanguageByName(String name) {
        return Optional.ofNullable(languagesByName.get(name.toLowerCase()));
    }

    private static Map<String, String> makeHeaders(@Nullable String language, @Nullable String tvmUserTicket) {
        var result = new HashMap<String, String>();
        if (language != null) {
            result.put(HttpHeaderNames.ACCEPT_LANGUAGE, language);
        }
        if (!StringUtils.isEmpty(tvmUserTicket)) {
            result.put(TvmHeaders.USER_TICKET, tvmUserTicket);
        }
        return result;
    }

    public List<OrganizationInfo> getOrganizationsInfo(Collection<Long> permalinkIds,
                                                       Language language,
                                                       @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        return getOrganizationsInfo(permalinkIds, language, tvmUserTicket, DEFAULT_FIELDS);
    }

    /**
     * Получение данных об организациях по списку пермалинков и языку интерфейса.
     * Ответ приведен к виду, используемому интерфейсом
     *
     * @param permalinkIds    Список пермалинков организаций
     * @param language        Язык интерфейса
     * @param tvmUserTicket   Пользовательский TVM тикет. Обычно получается на этапе аутентификации из BlackBox.
     *                        Может быть {@link Nullable} если идет запрос из интапи для perl-директа.
     * @param requestedFields набор запрашиваемых у Справочника полей организации
     * @return Список данных об организациях
     */
    public List<OrganizationInfo> getOrganizationsInfo(Collection<Long> permalinkIds,
                                                       Language language,
                                                       @Nullable String tvmUserTicket,
                                                       Collection<String> requestedFields)
            throws OrganizationsClientException {
        permalinkIds = filterAndMapToSet(permalinkIds, Objects::nonNull, identity());
        if (permalinkIds.isEmpty()) {
            return emptyList();
        }

        // Структуры практически идентичны, но в схеме нет наследования, потому копипаста
        String lang = language.name().toLowerCase();
        if (permalinkIds.size() == 1) {
            Long permalinkId = permalinkIds.iterator().next();
            PubApiCompanyData company = getSingleOrganizationInfo(permalinkId,
                    requestedFields, lang, tvmUserTicket);
            return company == null ? emptyList() :
                    List.of(OrganizationInfoConverters.convertSingleOrganizationInfo(permalinkId, lang, company));
        } else {
            PubApiCompaniesData companies =
                    getMultipleOrganizationsInfo(permalinkIds, requestedFields, lang, tvmUserTicket);
            return OrganizationInfoConverters.convertMultipleOrganizationsInfo(companies, lang);
        }
    }

    /**
     * Получение данных по нескольким организациям.
     * Возвращает необработанные значения. Для обработки можно воспользоваться методами из
     * {@link OrganizationInfoConverters}
     */
    @Nonnull
    public PubApiCompaniesData getMultiplePublishedOrganizationsInfo(Collection<Long> permalinkIds,
                                                                     Collection<String> fields,
                                                                     String language,
                                                                     @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        Set<String> fieldSet = new HashSet<>(fields);
        fieldSet.add(PubApiExtraCompanyField.PUBLISHING_STATUS.getValue());
        PubApiCompaniesData companies =
                getMultipleOrganizationsInfo(permalinkIds, emptyList(), fieldSet, language, tvmUserTicket, ANY);
        companies.getCompanies().removeIf(c -> c == null || c.getPublishingStatus() != PublishingStatus.PUBLISH);
        return companies;
    }

    /**
     * Получение данных по нескольким организациям.
     * Возвращает список uid с доступом к организации для каждого permalinkId
     */
    @Nonnull
    public Map<Long, Collection<Long>> getOrganizationsUidsWithModifyPermission(
            Collection<Long> permalinkIds, Collection<Long> uids, String language, @Nullable String tvmUserTicket) {
        permalinkIds = filterAndMapToSet(permalinkIds, Objects::nonNull, identity());
        if (permalinkIds.isEmpty()) {
            return emptyMap();
        }
        PubApiCompaniesData companies = getMultipleOrganizationsInfo(
                permalinkIds, uids, Set.of(GRANTS_FIELD), language, tvmUserTicket, ANY);
        return listToMap(companies.getCompanies(), PubApiCompany::getId, PubApiCompany::getUidsWithModifyPermission);
    }

    /**
     * Получение данных по нескольким организациям.
     * Возвращает необработанные значения.
     * Для обработки можно воспользоваться методами из {@link OrganizationInfoConverters}
     */
    @Nonnull
    public PubApiCompaniesData getMultipleOrganizationsInfo(Collection<Long> permalinkIds,
                                                            Collection<String> fields,
                                                            String language,
                                                            @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        return getMultipleOrganizationsInfo(permalinkIds, emptyList(), fields, language, tvmUserTicket, ANY);
    }

    /**
     * Получение данных по собственным организациям.
     * Возвращает необработанные значения.
     * Для обработки можно воспользоваться методами из {@link OrganizationInfoConverters}
     */
    @Nonnull
    public PubApiCompaniesData getMultipleOwnOrganizationsInfo(Collection<Long> uids,
                                                               Collection<String> fields,
                                                               String language,
                                                               @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        return getMultipleOrganizationsInfo(emptyList(), uids, fields, language, tvmUserTicket, OWNED);
    }

    /**
     * Поиск организации по тексту
     * Возвращает необработанные значения.
     * Для обработки можно воспользоваться методами из {@link OrganizationInfoConverters}
     */
    @Nonnull
    public List<OrganizationInfo> getMultipleOrganizationsInfoByText(String text,
                                                                     Language language,
                                                                     @Nullable String tvmUserTicket,
                                                                     Set<PubApiExtraCompanyField> requestedFields)
            throws OrganizationsClientException {
        if (StringUtils.isBlank(text)) {
            return emptyList();
        }
        PubApiCompanySearchBody searchBody = new PubApiCompanySearchBody();
        PubApiCompanySearchFilter searchFilter = new PubApiCompanySearchFilter();
        searchFilter.setText(text);
        searchFilter.setUserAccessLevel(ANY);
        searchBody.setFilter(searchFilter);
        searchBody.setExpand(new ArrayList<>(requestedFields));

        PubApiCompaniesData result = getMultipleOrganizationsInfoAsPost(searchBody, language, tvmUserTicket);

        if (result.getCompanies() != null) {
            List<PubApiCompany> pubApiCompanies = new ArrayList<>(result.getCompanies());
            pubApiCompanies.removeIf(c -> c == null || c.getPublishingStatus() != PublishingStatus.PUBLISH);
            result.setCompanies(pubApiCompanies);
        }

        return OrganizationInfoConverters.convertMultipleOrganizationsInfo(result, language.name().toLowerCase());
    }

    /**
     * Получение данных по одной организации. Возвращает необработанные значения.
     * Для обработки можно воспользоваться методами из {@link OrganizationInfoConverters}
     */
    @Nullable
    public PubApiCompanyData getSingleOrganizationInfo(
            Long permalinkId,
            Collection<String> fields,
            String language,
            @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        if (fields.contains(PubApiExtraCompanyField.PUBLISHING_STATUS.getValue())
                && !fields.contains(PubApiExtraCompanyField.HEAD_PERMALINK.getValue())) {
            fields = new HashSet<>(fields);
            fields.add(PubApiExtraCompanyField.HEAD_PERMALINK.getValue());
        }
        var headers = makeHeaders(language, tvmUserTicket);
        var call = organizationsApi.getSingleCompanyByPermalink(permalinkId, fields, headers);
        logRequest(call.getRequest().getAHCRequest());
        var serviceResponse = call.execute();
        PubApiCompanyData result = serviceResponse.getSuccess();
        if (result == null) {
            processErrors(serviceResponse.getErrors());
            return null;
        }
        logResponse(result);
        if (fields.contains(PubApiExtraCompanyField.LK_LINK.getValue())) {
            updateLink(result);
        }
        if (fields.contains(PubApiExtraCompanyField.METRIKA_DATA.getValue())) {
            updateMetrikaDataIfNeeded(result);
        }
        if (fields.contains(PubApiExtraCompanyField.PUBLISHING_STATUS.getValue())) {
            updatePublishingStatus(result);
        }
        return result;
    }

    /**
     * Получение данных по нескольким организациям. Возвращает необработанные значения.
     * Для обработки можно воспользоваться методами из {@link OrganizationInfoConverters}
     */
    @Nonnull
    protected PubApiCompaniesData getMultipleOrganizationsInfo(Collection<Long> permalinkIds,
                                                               Collection<Long> uids,
                                                               Collection<String> fields,
                                                               String language,
                                                               @Nullable String tvmUserTicket,
                                                               PubApiUserAccessLevel accessLevel)
            throws OrganizationsClientException {
        if (permalinkIds.isEmpty() && uids.isEmpty()) {
            return new PubApiCompaniesData();
        }

        // Если запрашиваем PUBLISHING_STATUS, то надо запросить и HEAD_PERMALINK,
        // чтобы выявить статус DUPLICATE
        boolean addToRequestHeadPermalink = fields.contains(PubApiExtraCompanyField.PUBLISHING_STATUS.getValue())
                && !fields.contains(PubApiExtraCompanyField.HEAD_PERMALINK.getValue());
        // Если в запросе было несколько пермалинков, то нужно запросить список дублей DUPLICATES,
        // чтобы можно было восстановить дубли по оригиналам.
        boolean addToRequestDuplicates = permalinkIds.size() > 1
                && !fields.contains(PubApiExtraCompanyField.DUPLICATES.getValue());
        if (addToRequestHeadPermalink || addToRequestDuplicates) {
            fields = new HashSet<>(fields);
            if (addToRequestHeadPermalink) {
                fields.add(PubApiExtraCompanyField.HEAD_PERMALINK.getValue());
            }
            if (addToRequestDuplicates) {
                fields.add(PubApiExtraCompanyField.DUPLICATES.getValue());
            }
        }
        var headers = makeHeaders(language, tvmUserTicket);
        HashSet<Long> uniquePermalinkIds = new HashSet<>(permalinkIds);
        var callsPack =
                organizationsApi.getCompaniesByPermalinkAndUid(uniquePermalinkIds, uids, fields, accessLevel, headers);

        callsPack.getRequests(permalinkChunkSize).forEach(request -> logRequest(request.getAHCRequest()));
        Collection<Result<PubApiCompaniesData>> results = callsPack.execute(permalinkChunkSize).values();
        StreamEx.of(results)
                .filter(result -> result.getSuccess() == null)
                .map(Result::getErrors)
                .forEach(OrganizationsClient::processErrors);
        List<PubApiCompaniesData> successfulResponses = StreamEx.of(results)
                .map(Result::getSuccess)
                .filter(Objects::nonNull)
                .toList();
        successfulResponses.forEach(OrganizationsClient::logResponse);
        PubApiCompaniesData summaryResult = mergePubApiCompaniesData(successfulResponses);

        if (fields.contains(PubApiExtraCompanyField.LK_LINK.getValue())) {
            updateLink(summaryResult);
        }
        if (fields.contains(PubApiExtraCompanyField.METRIKA_DATA.getValue())) {
            updateMetrikaDataIfNeeded(summaryResult);
        }
        if (fields.contains(PubApiExtraCompanyField.DUPLICATES.getValue())) {
            resolveDuplicates(summaryResult, uniquePermalinkIds);
        }
        if (fields.contains(PubApiExtraCompanyField.PUBLISHING_STATUS.getValue())) {
            updatePublishingStatus(summaryResult);
        }
        return summaryResult;
    }

    /**
     * Получение данных по нескольким организациям. Возвращает необработанные значения.
     * Для обработки можно воспользоваться методами из {@link OrganizationInfoConverters}
     */
    @Nonnull
    private PubApiCompaniesData getMultipleOrganizationsInfoAsPost(PubApiCompanySearchBody searchBody,
                                                                   Language language,
                                                                   @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        if (CollectionUtils.isEmpty(searchBody.getFilter().getPermalinks()) &&
                CollectionUtils.isEmpty(searchBody.getFilter().getUids()) &&
                StringUtils.isBlank(searchBody.getFilter().getText())) {
            return new PubApiCompaniesData();
        }

        List<PubApiExtraCompanyField> expand = searchBody.getExpand();
        // Если запрашиваем статус, то нужно запросить и головной пермалинк для обработки случая,
        //   когда запрашиваемый пермалинк - дубликат.
        if (expand != null
                && expand.contains(PubApiExtraCompanyField.PUBLISHING_STATUS)
                && !expand.contains(PubApiExtraCompanyField.HEAD_PERMALINK)) {
            searchBody.setExpand(new ArrayList<>(searchBody.getExpand()));
            searchBody.addExpandItem(PubApiExtraCompanyField.HEAD_PERMALINK);
        }

        var headers = makeHeaders(language.name().toLowerCase(), tvmUserTicket);
        var call = organizationsApi.getCompaniesByPermalinkAndUidAsPost(searchBody, headers);
        var serviceResponse = call.execute(searchByTextFetcherSettings);
        logRequest(call.getRequest().getAHCRequest());
        PubApiCompaniesData result = serviceResponse.getSuccess();
        if (result == null) {
            processErrors(serviceResponse.getErrors());
            return new PubApiCompaniesData();
        }
        logResponse(result);
        if (searchBody.getExpand().contains(PubApiExtraCompanyField.LK_LINK)) {
            updateLink(result);
        }
        if (searchBody.getExpand().contains(PubApiExtraCompanyField.METRIKA_DATA)) {
            updateMetrikaDataIfNeeded(result);
        }
        if (searchBody.getExpand().contains(PubApiExtraCompanyField.PUBLISHING_STATUS)) {
            updatePublishingStatus(result);
        }
        return result;
    }

    /**
     * @param request в запросе должны быть переданы новые стейты атрибутов, а не диффы с предыдущим вариантом
     */
    public void updateOrganization(
            @Nullable String tvmUserTicket,
            Long permalink,
            UpdateOrganizationRequest request
    ) {
        var headers = makeHeaders(null, tvmUserTicket);
        var call = organizationsApi.updateCompany(permalink, request, headers);
        logRequest(call.getRequest().getAHCRequest());
        var result = call.execute(writeRequestSettings);
        logResponse(result);

        var response = result.getSuccess();
        if (response == null) {
            throw new OrganizationsClientException("Can't update organization with permalink = " + permalink);
        }
    }

    @Nonnull
    public SpravResponse<PubApiCompany> createOnlineOrg(
            @Nullable String tvmUserTicket,
            CreateOnlineOrgRequest request
    ) {
        // Шаг 1: резервируем пермалинк
        long permalinkId = reservePermalink();

        // Шаг 2: создаём организацию
        var headers = makeHeaders(null, tvmUserTicket);
        PubApiCreateCompanyRequest createRequest = request.toPubApiCreateCompanyRequest(permalinkId);
        Call<SpravResponse<PubApiCompany>> call = organizationsApi.createCompany(createRequest, headers);
        logRequest(call.getRequest().getAHCRequest());
        Result<SpravResponse<PubApiCompany>> result = call.execute(writeRequestSettings);
        logResponse(result);

        SpravResponse<PubApiCompany> response = result.getSuccess();
        if (response == null) {
            processErrors(result.getErrors());
            throw new OrganizationsClientException("Can't create organization");
        }

        response.onSuccess((success) -> {
            // Шаг 3: добавляем счётчики Метрики, если успешно создали организацию
            updateMetrikaDataIfNeeded(success);
            return null;
        });

        return response;
    }

    private long reservePermalink() {
        Call<List<Long>> call = organizationsApi.reservePermalinks();
        logRequest(call.getRequest().getAHCRequest());

        Result<List<Long>> reservePermalinkResult = call.execute(writeRequestSettings);
        List<Long> reservedPermalinks = reservePermalinkResult.getSuccess();
        if (reservedPermalinks == null) {
            processErrors(reservePermalinkResult.getErrors());
            throw new OrganizationsClientException("Can't reserve permalinkId");
        }
        logResponse(reservePermalinkResult);

        checkState(!reservedPermalinks.isEmpty(), "Got empty permalink list from Sprav");
        return reservedPermalinks.get(0);
    }

    public List<MetrikaData> getOrganizationsCountersData(Collection<Long> permalinkIds,
                                                          String language,
                                                          @Nullable String tvmUserTicket)
            throws OrganizationsClientException {
        if (permalinkIds.isEmpty()) {
            return emptyList();
        }
        var headers = makeHeaders(language, tvmUserTicket);
        var call = organizationsApi.getOrganizationsCountersData(permalinkIds, headers);
        logRequest(call.getRequest().getAHCRequest());
        var serviceResponse = call.execute();
        List<MetrikaData> metrikaDataList = serviceResponse.getSuccess();
        if (metrikaDataList == null) {
            processErrors(serviceResponse.getErrors());

            return emptyList();
        }
        logResponse(metrikaDataList);
        Map<Long, MetrikaData> metrikaDataByPermalinkId = StreamEx.of(metrikaDataList)
                .toMap(MetrikaData::getPermalink, Function.identity());
        for (Long permalinkId : permalinkIds) {
            MetrikaData metrikaData = metrikaDataByPermalinkId.get(permalinkId);
            MetrikaData createdMetrikaData = createMetrikaDataIfNeeded(permalinkId, metrikaData);
            if (createdMetrikaData != null) {
                metrikaDataByPermalinkId.put(permalinkId, createdMetrikaData);
            }
        }
        metrikaDataList = new ArrayList<>(metrikaDataByPermalinkId.values());
        return metrikaDataList;
    }

    private void updateMetrikaDataIfNeeded(PubApiCompaniesData companies) {
        companies.getCompanies().forEach(company -> {
            MetrikaData metrikaData = createMetrikaDataIfNeeded(company.getId(), company.getMetrikaData());
            company.setMetrikaData(metrikaData);
        });
    }

    private void updateMetrikaDataIfNeeded(PubApiCompanyData company) {
        company.setMetrikaData(createMetrikaDataIfNeeded(company.getId(), company.getMetrikaData()));
    }

    private void updateMetrikaDataIfNeeded(PubApiCompany company) {
        company.setMetrikaData(createMetrikaDataIfNeeded(company.getId(), company.getMetrikaData()));
    }

    private MetrikaData createMetrikaDataIfNeeded(Long permalinkId,
                                                  @Nullable MetrikaData currentMetrikaData) {
        if (createMetrikaCounterProperty.getOrDefault(false)) {
            if (currentMetrikaData == null || StringUtils.isEmpty(currentMetrikaData.getCounter())) {
                MetrikaData createdMetrikaData = createCounter(permalinkId);
                if (createdMetrikaData != null) {
                    return createdMetrikaData;
                }
            }
        }
        return currentMetrikaData;
    }

    public MetrikaData createCounter(Long permalinkId)
            throws OrganizationsClientException {
        var call = organizationsApi.createCounterByPermalink(permalinkId);
        logRequest(call.getRequest().getAHCRequest());
        var serviceResponse = call.execute();
        MetrikaData metrikaData = serviceResponse.getSuccess();
        if (metrikaData == null) {
            processErrors(serviceResponse.getErrors());
            return null;
        }
        logResponse(metrikaData);
        return metrikaData;
    }

    /**
     * Обработка ошибок. Бросает {@link OrganizationsClientException} если ошибка более-менее известная.
     */
    private static void processErrors(List<Throwable> errors) throws OrganizationsClientException {
        for (Throwable throwable : errors) {
            logger.error("Failed to get company data", throwable);
            if (throwable instanceof ErrorResponseWrapperException) {
                ErrorResponseWrapperException e = (ErrorResponseWrapperException) throwable;
                if (e.getResponse() == null || KNOWN_ERROR_CODES.contains(e.getResponse().getStatusCode())) {
                    throw new OrganizationsClientException(e);
                }
            }
        }
    }

    /**
     * Логирование ответа.
     */
    private static void logResponse(Object response) {
        logger.info("Response company data {}", response);
    }

    /**
     * Логирование запроса.
     */
    private static void logRequest(Request request) {
        if (request == null) {
            return;
        }
        StringBuilder sb = new StringBuilder(request.getUrl());
        sb.append("\t")
                .append(request.getMethod());
        sb.append("\theaders:");
        if (!request.getHeaders().isEmpty()) {
            for (Map.Entry<String, String> header : request.getHeaders()) {
                String key = header.getKey();
                String value = TVM_TICKED_HEADERS.contains(key) ? hideSignature(header.getValue()) : header.getValue();
                sb.append("\t")
                        .append(key)
                        .append(":")
                        .append(value);
            }
        }
        if (isNonEmpty(request.getFormParams())) {
            sb.append("\tformParams:");
            for (Param param : request.getFormParams()) {
                sb.append("\t")
                        .append(param.getName())
                        .append(":")
                        .append(param.getValue());
            }
        }
        if (!StringUtils.isBlank(request.getStringData())) {
            sb.append("\tstringDataForPost:\t");
            sb.append(request.getStringData());
        }
        logger.info("Request to Sprav: {}", sb.toString());
    }

    private static String hideSignature(String tvmTicket) {
        if (tvmTicket == null) {
            return null;
        }
        int index = tvmTicket.lastIndexOf(':');
        if (index < 0) {
            return tvmTicket;
        }
        return tvmTicket.substring(0, index + 1) + "...";
    }

}
