package ru.yandex.direct.pdfgen;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.fop.apps.FOPException;
import org.apache.fop.apps.Fop;
import org.apache.fop.apps.FopConfParser;
import org.apache.fop.apps.FopFactory;
import org.apache.fop.apps.MimeConstants;
import org.apache.fop.apps.io.ResourceResolverFactory;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.context.Context;
import org.thymeleaf.templatemode.TemplateMode;
import org.thymeleaf.templateresolver.ClassLoaderTemplateResolver;
import org.xml.sax.SAXException;

/**
 * Создаватель PDF: обрабатывает thymeleaf-шаблон из ресурсов и с помощью Apache FOP
 * превращает это всё в PDF.
 * <p>
 * Apache FOP знает про какие-то шрифты Яндекса, так что можно писать по-русски, там
 * есть лигатуры. Лучше всего начать с копирования и вставки test.fo.thyme из тестовых
 * ресурсов этого модуля.
 * <p>
 * Чтобы этим воспользоваться, надо вызвать конструктор и на созданном объекте вызвать
 * buildPdf(). Конструктор может делать тяжёлые операции, так что лучше создать один
 * объект при инициализации приложения и им создавать PDF по запросу клиента.
 * <p>
 * buildPdf() поточнобезопасный, потому что synchronized; если надо делать много pdf
 * параллельно, потребителю предлагается сделать пул объектов PdfBuilder самостоятельно.
 * Нижележащий Apache FOP не обещает, что он поточнобезопасный,
 * {@see https://xmlgraphics.apache.org/fop/2.2/embedding.html#multithreading},
 * так что удалять синхронизацию в этом компоненте не рекомендуется.
 */
public class PdfBuilder {
    private static final String BASE_URI = "file:///";
    private final FopFactory fopFactory;
    private final TemplateEngine templateEngine;

    public PdfBuilder() {
        // templateEngine
        ClassLoaderTemplateResolver templateResolver = new ClassLoaderTemplateResolver();
        templateResolver.setTemplateMode(TemplateMode.XML);

        TemplateEngine engine = new TemplateEngine();
        engine.setTemplateResolver(templateResolver);

        this.templateEngine = engine;

        // fopFactory
        Map<String, String> fontPaths = new HashMap<>();
        fontPaths.put("YandexSans", ClassLoader.getSystemResource("YandexSans.ttf").toString());
        fontPaths.put("YandexSansBold", ClassLoader.getSystemResource("YandexSansBold.ttf").toString());

        Context context = new Context();
        context.setVariable("fontPaths", fontPaths);
        String fopConf = engine.process("fop.conf.thyme", context);

        try {
            FopConfParser fopConfParser = new FopConfParser(
                    new ByteArrayInputStream(fopConf.getBytes(StandardCharsets.UTF_8)),
                    new URI(BASE_URI),
                    ResourceResolverFactory.createDefaultResourceResolver());

            fopFactory = fopConfParser.getFopFactoryBuilder().build();
        } catch (SAXException | IOException | URISyntaxException e) {
            throw new FopFactoryInitializationException(e);
        }
    }

    /**
     * @param parameters объект параметров: из него с помощью {@link PropertyUtils#describe}
     *                   будет сделан <code>Map&lt;String, Object&gt;</code>, из которого
     *                   будет сделан thymeleaf-контекст, так что объект должен быть сделан
     *                   по JavaBeans: объект публичного класса с публичными геттерами
     *                   для всех полей
     * @throws IllegalArgumentException если parameters не удалось разобрать и превратить
     *                                  в <code>Map&lt;String, Object&gt;</code>
     * @throws PdfBuilderException      если синтаксис получившегося после шаблонизации xsl-fo неправильный
     */
    public synchronized byte[] buildPdf(String templateFilename, Object parameters) {
        Map<String, Object> parameterMap;

        try {
            parameterMap = PropertyUtils.describe(parameters);
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            // NB: если здесь вылетает исключение, надо проверить, что:
            // (1) класс, объектом которого является parameters, публично доступен
            // (2) у всех полей есть геттеры: у каждого field есть getField()
            // (3) эти геттеры публично доступны
            throw new IllegalArgumentException(e);
        }

        Context context = new Context(null, parameterMap);
        String foSource = templateEngine.process(templateFilename, context);

        InputStream fopInputStream = new ByteArrayInputStream(foSource.getBytes(StandardCharsets.UTF_8));
        Source src = new StreamSource(fopInputStream);

        ByteArrayOutputStream out = new ByteArrayOutputStream();

        try {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            Fop fop = fopFactory.newFop(MimeConstants.MIME_PDF, out);
            Result res = new SAXResult(fop.getDefaultHandler());
            transformer.transform(src, res);
        } catch (FOPException | TransformerException e) {
            throw new PdfBuilderException(e);
        }

        return out.toByteArray();
    }

    private static final class FopFactoryInitializationException extends RuntimeException {
        FopFactoryInitializationException(Throwable cause) {
            super(cause);
        }
    }
}
