package ru.yandex.direct.pokazometer;

import java.util.Arrays;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public class PhraseRequest {
    private Long keywordId;
    private String phrase;
    private Long price;

    /**
     * Создает объект фразы
     *
     * @param phrase    текст фразы
     * @param price     цена фразы в микродолларах. Используется для расчёта ContextCoverage
     * @param keywordId id фразы в директе. Не используется в запросе показометра.
     *                  Но если задан, то пробрасывается клиентом во PhraseResponse при получении ответа от показометра.
     */
    public PhraseRequest(String phrase, @Nullable Long price, @Nullable Long keywordId) {
        this.phrase = phrase;
        this.price = price;
        this.keywordId = keywordId;
    }

    public PhraseRequest(String phrase, @Nullable Long price) {
        this(phrase, price, null);
    }

    /**
     * Возвращает текст фразы, исключая из нее минус-слова
     *
     * @return текст фразы
     */
    public String getPhrase() {
        return Arrays.stream(phrase.split("\\s+")).filter(s -> !s.startsWith("-"))
                .collect(Collectors.joining(" "));
    }

    /**
     * Возвращает цену клика
     *
     * @return цена клика в микродолларах
     */
    @Nullable
    public Long getPrice() {
        return price;
    }

    /**
     * Возвращает идентификатор фразы
     */
    public Long getKeywordId() {
        return keywordId;
    }

    @Override
    public String toString() {
        return "PhraseRequest{" +
                "phrase='" + phrase + '\'' +
                ", price=" + price +
                '}';
    }
}
