package ru.yandex.direct.pokazometer;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

public class PhraseResponse {
    /**
     * Степени покрытия в процентах от общего количества кликов
     */
    public enum Coverage {
        /**
         * Низкая (20%) степень покрытия
         */
        LOW(20),
        /**
         * Средняя (50%) степень покрытия
         */
        MEDIUM(50),
        /**
         * Высокая (100%) степень покрытия
         */
        HIGH(100);

        private int percentage;

        Coverage(int percentage) {
            this.percentage = percentage;
        }

        public int getPercentage() {
            return percentage;
        }
    }

    private Long keywordId;
    private String phrase;
    private Long contextCoverage;
    private Map<Coverage, Long> priceByCoverage;
    private Map<Long, Integer> clicksByCost;

    /**
     * Создает объект фразы
     *
     * @param phrase текст фразы
     */
    private PhraseResponse(String phrase, Long keywordId) {
        this.phrase = phrase;
        this.keywordId = keywordId;
        this.priceByCoverage = new HashMap<>();
        this.clicksByCost = Collections.emptyMap();
    }

    /**
     * Возвращает текст фразы, исключая из нее минус-слова
     *
     * @return текст фразы
     */
    public String getPhrase() {
        return Arrays.stream(phrase.split("\\s+")).filter(s -> !s.startsWith("-"))
                .collect(Collectors.joining(" "));
    }

    /**
     * Возвращает прогноз покрытия для текущей цены
     *
     * @return прогноз покрытия в процентах от максимального
     */
    @SuppressWarnings("WeakerAccess")
    public Long getContextCoverage() {
        return contextCoverage;
    }

    /**
     * Устанавливает прогноз покрытия для текущей цены
     *
     * @param contextCoverage прогноз покрытия в процентах от максимального
     */
    public void setContextCoverage(Long contextCoverage) {
        this.contextCoverage = contextCoverage;
    }

    /**
     * Возвращает прогноз цены клика для указанного покрытия
     *
     * @param coverage уровень покрытия
     * @return цена клика в микродолларах
     */
    @SuppressWarnings("WeakerAccess")
    public Long getPriceByCoverage(Coverage coverage) {
        return priceByCoverage.get(coverage);
    }

    public Map<Coverage, Long> getPriceByCoverage() {
        return priceByCoverage;
    }

    /**
     * Устанавливает прогноз цены клика для указанного покрытия
     *
     * @param coverage уровень покрытия
     * @param price    цена клика в микродолларах
     */
    public void setPriceByCoverage(Coverage coverage, Long price) {
        priceByCoverage.put(coverage, price);
    }

    /**
     * Получить статистику кликов и ставок
     *
     * @return мапа: ставка - количество кликов
     */
    public Map<Long, Integer> getClicksByCost() {
        return clicksByCost;
    }

    /**
     * Установить все соответствия стомостей клика к показам
     *
     * @param clicksByCost - мапа соответствия стоимости клика и показов
     */
    public void setClicksByCost(Map<Long, Integer> clicksByCost) {
        this.clicksByCost = clicksByCost;
    }

    /**
     * Идентификатор ключевой фразы в директе.
     * Не возвращается показометром, но устанавливается клиентом, в случае если id был задан в соответсвующем запросе PhraseRequest
     *
     * @return - id ключевой фразы
     */
    @Nullable
    public Long getKeywordId() {
        return keywordId;
    }

    /**
     * Установить id ключевой фразы
     *
     * @param keywordId - id ключевой фразы
     */
    public void setKeywordId(Long keywordId) {
        this.keywordId = keywordId;
    }

    public static PhraseResponse on(PhraseRequest request) {
        return new PhraseResponse(request.getPhrase(), request.getKeywordId());
    }


    @Override
    public String toString() {
        return "PhraseResponse{" +
                "phrase='" + phrase + '\'' +
                ", contextCoverage=" + contextCoverage +
                ", priceByCoverage=" + priceByCoverage +
                '}';
    }
}
