package ru.yandex.direct.pokazometer;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.pokazometer.internal.ClicksDistributionParams;
import ru.yandex.direct.pokazometer.internal.ClicksDistributionResponse;
import ru.yandex.direct.pokazometer.internal.ClicksDistributionResult;
import ru.yandex.direct.pokazometer.internal.CostData;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonRpcRequest;
import ru.yandex.direct.utils.math.MathUtils;
import ru.yandex.direct.utils.math.Point;

import static java.util.Arrays.asList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.ListUtils.resample;

/**
 * Клиент для показометра
 * <a href="https://wiki.yandex-team.ru/pokazometer/">Описание Показометр</a>
 */
public class PokazometerClient {
    /**
     * Так как показометр может присылать очень много точек, для снижения нагрузки берется
     * только RESAMPLE_SIZE точек, равномерно распределенных среди исходных -
     * первая, последняя, и оставшиеся между ними через равные промежутки
     */
    private static final int RESAMPLE_SIZE = 5;

    /**
     * Набор раасчитываемых покрытий в процентах от максимального количества кликов -
     * низкое (20%), среднее (50%) и высокое (100%)
     */
    private static final List<PhraseResponse.Coverage> COVERAGES =
            asList(PhraseResponse.Coverage.LOW, PhraseResponse.Coverage.MEDIUM, PhraseResponse.Coverage.HIGH);

    private static final Logger logger = LoggerFactory.getLogger(PokazometerClient.class);

    private final ParallelFetcherFactory fetcherFactory;
    private final String pokazometerUrl;

    /**
     * Создает экземпляр клиента для Показометра
     */
    public PokazometerClient(ParallelFetcherFactory fetcherFactory, String pokazometerUrl) {
        this.fetcherFactory = fetcherFactory;
        this.pokazometerUrl = pokazometerUrl;
    }

    /**
     * Обсчитывает список групп, и возвращает все фразы из них с посчитанным покрытием из указанной
     * цены клика, а также ценой клика для трех вариантов покрытия - низкое (20%), среднее (50%)
     * и высокое (100%)
     *
     * @param groups Список групп для обсчета
     * @return Обсчитанные фразы из групп
     */
    public IdentityHashMap<GroupRequest, GroupResponse> get(List<GroupRequest> groups) {
        logger.trace("get {}", groups);
        List<JsonRpcRequest<ClicksDistributionParams>> requests = new ArrayList<>(groups.size());
        int id = 1;
        for (GroupRequest group : groups) {
            requests.add(createRequest(group, id));
            id++;
        }

        List<ClicksDistributionResult> innerResults = pokazometerGet(requests);

        IdentityHashMap<GroupRequest, GroupResponse> results = new IdentityHashMap<>(groups.size());
        for (int i = 0; i < groups.size(); i++) {
            results.put(groups.get(i), processResult(innerResults.get(i), groups.get(i)));
        }
        return results;
    }

    private JsonRpcRequest<ClicksDistributionParams> createRequest(GroupRequest group, int id) {
        ClicksDistributionParams params = new ClicksDistributionParams(
                mapList(group.getPhrases(), PhraseRequest::getPhrase),
                group.getGeo());
        return new JsonRpcRequest<>("ClicksDistribution", params, id);
    }

    private GroupResponse processResult(ClicksDistributionResult result, GroupRequest group) {
        if (!result.isSuccessful()) {
            return GroupResponse.failure(result.getErrors());
        }
        List<PhraseRequest> requests = group.getPhrases();
        List<PhraseResponse> responses = new ArrayList<>(requests.size());
        for (int i = 0; i < requests.size(); i++) {
            PhraseRequest request = requests.get(i);
            PhraseResponse response = PhraseResponse.on(request);
            List<CostData> prices = getPrices(result, i);
            if (!prices.isEmpty()) {
                // есть из чего интерполировать, показометр может вернуть пустой список цен
                calculateContextCoverage(request, response, prices);
                calculateCoveragePrice(response, prices);
                fillAllCostsAndClicks(response, prices);
            }
            responses.add(response);
        }

        return GroupResponse.success(responses);
    }

    private List<ClicksDistributionResult> pokazometerGet(List<JsonRpcRequest<ClicksDistributionParams>> requests) {
        try (TraceProfile profile =
                     Trace.current().profile("pokazometer:_request_pokazometer_data_parallel", "", requests.size())) {
            List<ParsableRequest<String>> asyncRequests = mapList(requests,
                    request -> {
                        RequestBuilder builder = new RequestBuilder("POST")
                                .setUrl(pokazometerUrl)
                                .setCharset(Charset.forName("UTF-8"))
                                .setBody(request.toString())
                                .addHeader("Content-Type", "application/json");
                        Request innerRequest = builder.build();
                        return new ParsableStringRequest(request.getId(), innerRequest);
                    });

            ParallelFetcher<String> parallelFetcher = fetcherFactory.getParallelFetcherWithMetricRegistry(
                    SolomonUtils.getParallelFetcherMetricRegistry(profile.getFunc()));
            Map<Long, Result<String>> resultMap = parallelFetcher.execute(asyncRequests);
            List<Result<String>> results = mapList(requests, request -> resultMap.get(request.getId().longValue()));
            return mapList(results, result -> result.getSuccess() != null
                    ? fromJson(result.getSuccess(), ClicksDistributionResponse.class).getResult()
                    : ClicksDistributionResult.failure(result.getErrors()));
        } catch (InterruptedException ex) {
            logger.info("Requests {} were unexpectedly interrupted.", requests);
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    private List<CostData> getPrices(ClicksDistributionResult result, int index) {
        List<CostData> prices = result.getDistribution().get(index);
        prices.sort(Comparator.comparingInt(CostData::getCount));
        return resample(prices, RESAMPLE_SIZE);
    }

    private void calculateContextCoverage(PhraseRequest request, PhraseResponse phrase, List<CostData> prices) {
        if (request.getPrice() == null) {
            // пропускаем заполнение ContextCoverage, если для фразы не задана цена
            return;
        }
        double price = request.getPrice();
        phrase.setContextCoverage(Math.round(
                MathUtils.interpolateLinear(price, getCoverageOfPricePoints(prices)).getY() * 100));
    }

    private void calculateCoveragePrice(PhraseResponse phrase, List<CostData> prices) {
        List<Point> points = getPriceOfCoveragePoints(prices);
        for (PhraseResponse.Coverage coverage : COVERAGES) {
            phrase.setPriceByCoverage(coverage, Math.round(
                    MathUtils.interpolateLinear(coverage.getPercentage() / 100.0, points).getY()));
        }
    }

    private void fillAllCostsAndClicks(PhraseResponse phrase, List<CostData> prices) {
        phrase.setClicksByCost(StreamEx.of(prices)
                .mapToEntry(CostData::getCost, CostData::getCount)
                .toMap(Integer::max));
    }

    private List<Point> getCoverageOfPricePoints(List<CostData> prices) {
        double maxClicks = prices.get(prices.size() - 1).getCount();
        return mapList(prices, cd -> Point.fromDoubles(cd.getCost(), cd.getCount() / maxClicks));
    }

    private List<Point> getPriceOfCoveragePoints(List<CostData> prices) {
        double maxClicks = prices.get(prices.size() - 1).getCount();
        return mapList(prices, cd -> Point.fromDoubles(cd.getCount() / maxClicks, cd.getCost()));
    }
}
