package ru.yandex.direct.process;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;

import com.google.common.collect.Lists;
import org.yaml.snakeyaml.util.ArrayUtils;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.Interrupts;

import static java.util.Collections.emptyList;

public class Docker {
    public static final String DEFAULT_DOCKER_EXECUTABLE = "docker";
    private static final Charset CHARSET = StandardCharsets.UTF_8;

    private final String executable;

    public Docker() {
        this(DEFAULT_DOCKER_EXECUTABLE);
    }

    public Docker(String executable) {
        this.executable = executable;
    }

    public ProcessBuilder getProcessBuilder(String... args) {
        return new ProcessBuilder(Lists.asList(executable, args));
    }

    public void checkCall(String... args) throws InterruptedException {
        Processes.checkCall(Lists.asList(executable, args).toArray(args));
    }

    public String checkOutput(String... args) throws InterruptedException {
        ProcessCommunicator.StringReader stdoutReader = new ProcessCommunicator.StringReader(CHARSET);
        ProcessCommunicator.StringReader stderrReader = new ProcessCommunicator.StringReader(CHARSET);

        try (ProcessCommunicator communicator = capturedStdoutStderr(
                getProcessBuilder(args), stdoutReader, stderrReader
        )) {
            communicator.waitFor();
        } catch (ProcessExitStatusException exc) {
            throw new DockerException(stderrReader.getContent(), exc);
        }

        return stdoutReader.getContent();
    }

    public String checkOutputUninterruptibly(Duration timeout, String... args) {
        ProcessCommunicator.StringReader stdoutReader = new ProcessCommunicator.StringReader(CHARSET);
        ProcessCommunicator.StringReader stderrReader = new ProcessCommunicator.StringReader(CHARSET);

        try (ProcessCommunicator communicator = capturedStdoutStderr(
                getProcessBuilder(args), stdoutReader, stderrReader
        )) {
            Interrupts.criticalTimeoutWait(timeout, communicator::waitFor);
        } catch (ProcessExitStatusException exc) {
            throw new DockerException(stderrReader.getContent(), exc);
        }

        return stdoutReader.getContent();
    }

    public void checkContainerId(DockerContainerId expectedContainerId, String... command)
            throws InterruptedException {
        String outputContainerId = checkOutput(command).trim();
        if (!expectedContainerId.toString().equals(outputContainerId)) {
            throw new IllegalStateException(
                    Arrays.toString(command) + " returned unexpected container id: " + outputContainerId
                            + ", expected: " + expectedContainerId
            );
        }
    }

    public void checkContainerIdUninterruptibly(
            Duration timeout,
            DockerContainerId expectedContainerId,
            String... command
    ) {
        String outputContainerId = checkOutputUninterruptibly(timeout, command).trim();
        if (!expectedContainerId.toString().equals(outputContainerId)) {
            throw new IllegalStateException(
                    Arrays.toString(command) + " returned unexpected container id: " + outputContainerId
                            + ", expected: " + expectedContainerId
            );
        }
    }

    public Processes.CommunicationResult communicate(Duration timeout, String... args) {
        ProcessCommunicator.StringReader stdoutReader = new ProcessCommunicator.StringReader(CHARSET);
        ProcessCommunicator.StringReader stderrReader = new ProcessCommunicator.StringReader(CHARSET);

        try (ProcessCommunicator communicator = capturedStdoutStderr(
                getProcessBuilder(args), stdoutReader, stderrReader
        )) {
            Interrupts.criticalTimeoutWait(timeout, communicator::waitFor);
        } catch (ProcessExitStatusException exc) {
            throw new DockerException(truncatedString(stderrReader.getContent()), exc);
        }

        return new Processes.CommunicationResult(stdoutReader.getContent(), stderrReader.getContent());
    }

    public boolean isAvailable() throws InterruptedException {
        return Processes.commandIsExecutable(executable, "info");
    }

    public void push(String image) throws InterruptedException {
        Processes.checkCall(executable, "push", image);
    }

    /**
     * Определение поддержки IPv6 путём запуска Docker-контейнера.
     *
     * @param image Название или id любого Docker-образа. Выгодно подставлять те образы, которые уже есть на диске.
     * @return {@code true} если поддерживается IPv6, {@code false} если не поддерживается.
     */
    public boolean isDockerSupportsIpv6(String image) {
        Processes.CommunicationResult result = communicate(Duration.ofSeconds(5),
                "run", "--rm", "--entrypoint=cat", image,
                "/proc/sys/net/ipv6/conf/all/disable_ipv6");
        switch (result.getStdout().trim()) {
            case "0":
                return true;
            case "1":
                return false;
            default:
                throw new IllegalArgumentException(String.format(
                        "Unexpected result from image %s%nSTDOUT:%n%s%nSTDERR:%n%s%n", image,
                        truncatedString(result.getStdout()), truncatedString(result.getStderr())));
        }
    }

    private static ProcessCommunicator capturedStdoutStderr(
            ProcessBuilder processBuilder,
            Consumer<InputStream> stdoutReader,
            Consumer<InputStream> stderrReader
    ) {
        return new ProcessCommunicator.Builder(processBuilder)
                .withStdoutReader(stdoutReader)
                .withStderrReader(stderrReader)
                .build();
    }

    private static String truncatedString(String string) {
        if (string.length() < 10000) {
            return string;
        } else {
            return string.substring(0, 10000) + "... <truncated>";
        }
    }

    /**
     * Ищет запущенные контейнеры по названию
     * (точнее, по префиксу, так что в результате может присутствовать более 1 контейнера)
     *
     * @return id найденных контейнеров
     */
    public List<String> findContainersByName(String name, Duration timeout) {
        String ids = checkOutputUninterruptibly(timeout, "ps", "-a", "-f", "name=" + name, "--format={{.ID}}");
        return ids.isEmpty() ? emptyList() : ArrayUtils.toUnmodifiableList(ids.split("\n"));
    }

    /**
     * Останавливает и удаляет контейнер с указанным названием (если такой имеется)
     */
    public void stopAndRemoveContainerByName(String name) {
        List<String> ids = findContainersByName(name, Duration.ofSeconds(20));
        if (ids.isEmpty()) {
            return;
        }
        try {
            checkCall("stop", name);
        } catch (ProcessExitStatusException e) {
            // Вероятно, контейнер с таким названием уже был остановлен
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
        try {
            checkCall("rm", name);
        } catch (ProcessExitStatusException e) {
            // Вероятно, контейнер с таким названием уже был удалён
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }
}
