package ru.yandex.direct.process;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.HostPort;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;

import static org.apache.commons.lang3.StringUtils.isEmpty;

public class DockerContainer implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(DockerContainer.class);

    private static final Duration CLOSE_TIMEOUT = Duration.ofSeconds(10);
    private static final Duration CLOSE_CHECK_TIMEOUT = Duration.ofSeconds(2);

    private static final String HEALTHY_STATUS = "healthy";
    private static final Duration DEFAULT_WAIT_CONTAINER_HEALTHY_DURATION = Duration.ofSeconds(180);
    private static final Duration DEFAULT_SLEEP_BETWEEN_HEALTHY_CHECK = Duration.ofSeconds(1);
    private static final Duration DEFAULT_INSPECT_HEALTHY_STATUS_TIMEOUT = Duration.ofSeconds(5);

    private final DockerContainerId containerId;
    private final DockerRunner runner;
    private boolean exitedAndErrorReported;

    public DockerContainer(DockerRunner runner) throws InterruptedException {
        this.exitedAndErrorReported = false;
        this.runner = runner.copy();
        this.containerId = runner.run();
    }

    public DockerContainerId getContainerId() {
        return containerId;
    }

    public Docker getDocker() {
        return runner.getDocker();
    }

    public String getImage() throws InterruptedException {
        return getDocker().checkOutput(
                "inspect",
                "--type=container",
                "--format={{.Config.Image}}",
                this.containerId.toString()
        ).trim();
    }

    /**
     * @return {@code hostname}, заданный при старте контейнера через параметр {@code --hostname}
     */
    public String getHostname() {
        return runner.getHostname().orElseThrow(() ->
                new IllegalStateException("Host name is not specified for " + this));
    }

    /**
     * Возвращает host, где бегут docker-контейнеры.
     * <p/>
     * В качестве эвристики смотрим на переменную окружения {@code DOCKER_HOST}.
     * На неё ориентируется процесс docker CLI. Если она задана, команды будут отправляться в демон на указанном хосте.
     * <p/>
     * Уважаем эту переменную при получении адреса запущенного контейнера.
     * Выглядит она примерно так: {@code "ssh://user@host"}
     */
    private Optional<String> getDockerHost() {
        String dockerHost = System.getenv("DOCKER_HOST");
        if (dockerHost != null) {
            try {
                URI uri = new URI(dockerHost);
                return Optional.of(uri.getHost());
            } catch (URISyntaxException e) {
                throw new IllegalArgumentException(
                        "Unexpected value for env DOCKER_HOST. Expected: 'ssh://user@host'. Actual: " + dockerHost, e);
            }
        } else {
            return Optional.empty();
        }
    }

    /**
     * Возвращает {@link HostPort}, где мета-адрес {@code 0.0.0.0} заменён на более подходящий.
     * Например, {@code 127.0.0.1} для локального docker-демона или имя хоста для удалённого docker'а.
     * <p/>
     * {@code 0.0.0.0} появляется как адрес, на котором сервис слушает подключения.
     * Попытки подключения к этому адресу не всегда успешны.
     */
    private HostPort fixLocalHost(HostPort exposedPort) {
        if (!"0.0.0.0".equals(exposedPort.getHost())) {
            return exposedPort;
        }
        String host = getDockerHost().orElse("127.0.0.1");
        return exposedPort.withHost(host);
    }

    public HostPort getPublishedPort(int port) throws InterruptedException {
        // Под макосью работает, как минимум, начиная с Docker 1.12.0-rc2-beta17
        String dockerPublishedPorts = getDocker().checkOutput(
                "port",
                containerId.toString(),
                String.valueOf(port)
        ).trim();
        // Если контейнер слушает порт на нескольких интерфейсах (например, 0.0.0.0 и ::), выберем первый
        if (dockerPublishedPorts.contains("\n")) {
            dockerPublishedPorts = dockerPublishedPorts.split("\n")[0];
        }
        HostPort exposedPort = HostPort.fromString(dockerPublishedPorts);
        return fixLocalHost(exposedPort);
    }

    public void copyFromContainer(Path src, Path dst) throws InterruptedException {
        getDocker().checkCall("cp", "--follow-link", containerId + ":" + src, dst.toString());
    }

    public String inspect(Duration timeout, String formatOutput) {
        return getDocker().checkOutputUninterruptibly(timeout, "inspect", "--type", "container", "-f", formatOutput,
                containerId.toString());
    }

    public JsonNode inspect(Duration timeout) {
        return JsonUtils.fromJson(getDocker().checkOutputUninterruptibly(
                timeout, "inspect", "--type", "container", containerId.toString()
        ));
    }

    public String commit(String tag) throws InterruptedException {
        String imageId = getDocker().checkOutput("commit", containerId.toString(), tag);
        if (imageId.startsWith("sha256:")) {
            return imageId.replaceFirst("sha256:", "").trim();
        } else {
            throw new IllegalStateException("Unexpected output from docker commit: " + imageId);
        }
    }

    public void start() throws InterruptedException {
        getDocker().checkContainerId(containerId, "start", containerId.toString());
    }

    public void stop() throws InterruptedException {
        getDocker().checkContainerId(containerId, "stop", containerId.toString());
    }

    /**
     * Отправляет SIGTERM главному процессу в контейнере, а по истечении {@code timeout} SIGKILL.
     */
    public void stop(Duration timeout) {
        getDocker().checkContainerIdUninterruptibly(timeout,
                containerId, "stop", "--time=" + Long.toString(timeout.getSeconds()), containerId.toString());
    }

    public void kill(String signal) throws InterruptedException {
        getDocker().checkContainerId(containerId, "kill", "--signal", signal, containerId.toString());
    }

    public void term() throws InterruptedException {
        kill("SIGTERM");
    }

    public String readStderr(Tail tail, Duration timeout) {
        if (tail instanceof Tail.Lines && ((Tail.Lines) tail).getCount() <= 0) {
            return "";
        }
        return getDocker().communicate(
                timeout,
                "logs",
                "--tail", tail.toString(),
                containerId.toString()
        ).getStderr();
    }

    public String readStdout(Tail tail, Duration timeout) {
        if (tail instanceof Tail.Lines && ((Tail.Lines) tail).getCount() <= 0) {
            return "";
        }
        return getDocker().communicate(
                timeout,
                "logs",
                "--tail", tail.toString(),
                containerId.toString()
        ).getStdout();
    }

    @SuppressWarnings("squid:S1166") // Ловим и проглатываем CriticalTimeoutExpired
    public Optional<String> tryReadStderr(Tail tail, Duration timeout) {
        try {
            return Optional.of(readStderr(tail, timeout));
        } catch (Interrupts.CriticalTimeoutExpired exc) {
            return Optional.empty();
        }
    }

    public Optional<String> tryReadStderr(Tail tail) {
        return tryReadStderr(tail, Duration.ofSeconds(5));
    }

    public void exec(String... cmd) throws InterruptedException {
        getDocker().checkCall(Lists.asList("exec", containerId.toString(), cmd).toArray(cmd));
    }

    public int waitAndReturnExitCode() throws InterruptedException {
        return Integer.parseInt(getDocker().checkOutput("wait", containerId.toString()).trim());
    }

    public void waitAndCheckExitCode() throws InterruptedException {
        int exitCode = waitAndReturnExitCode();
        if (exitCode != 0) {
            throw new DockerContainerExitStatusException(
                    containerId,
                    runner.getImage(),
                    runner.getCommandLine(),
                    exitCode,
                    tryReadStderr(new Tail.Lines(100)).orElse("<not fetched>")
            );
        }
    }

    protected void checkExitCode(Duration timeout) {
        MonotonicTime deadline = NanoTimeClock.now().plus(timeout);
        try {
            JsonNode info = inspect(timeout).get(0);
            JsonNode state = info.get("State");
            if ("exited".equals(state.get("Status").textValue()) && state.get("ExitCode").intValue() != 0) {
                throw new DockerContainerExitStatusException(
                        containerId,
                        runner.getImage(),
                        runner.getCommandLine(),
                        state.get("ExitCode").intValue(),
                        tryReadStderr(new Tail.Lines(100), deadline.minus(NanoTimeClock.now()))
                                .orElse("<not fetched>")
                );
            }
        } catch (RuntimeException exc) {
            // Устанавливаем флажок при любом исключении, а не только при DockerContainerExitStatusException
            exitedAndErrorReported = true;
            throw exc;
        }
    }

    @Override
    public void close() {
        MonotonicTime deadline = NanoTimeClock.now().plus(CLOSE_TIMEOUT);
        try {
            if (!exitedAndErrorReported) {
                checkExitCode(CLOSE_CHECK_TIMEOUT);
            }
        } finally {
            getDocker().checkContainerIdUninterruptibly(
                    deadline.minus(NanoTimeClock.now()),
                    containerId,
                    "rm", "--force", "--volumes", containerId.toString()
            );
        }
    }

    public static abstract class Tail {
        public static final All ALL = new All();

        public static Lines lines(long count) {
            return new Lines(count);
        }

        public static class All extends Tail {
            @Override
            public String toString() {
                return "ALL";
            }
        }

        public static class Lines extends Tail {
            private long count;

            public Lines(long count) {
                this.count = count;
            }

            public long getCount() {
                return count;
            }

            @Override
            public String toString() {
                return Long.toString(count);
            }
        }
    }

    public void waitUntilContainerBecomeHealthy() {
        waitUntilContainerBecomeHealthy(DEFAULT_WAIT_CONTAINER_HEALTHY_DURATION,
                DEFAULT_INSPECT_HEALTHY_STATUS_TIMEOUT, DEFAULT_SLEEP_BETWEEN_HEALTHY_CHECK);
    }

    public void waitUntilContainerBecomeHealthy(Duration waitContainerHealthyDuration,
                                                Duration inspectHealthyStatusTimeout,
                                                Duration sleepBetweenHealthyCheck) {
        NanoTimeClock clock = new NanoTimeClock();
        MonotonicTime deadline = clock.getTime().plus(waitContainerHealthyDuration);
        while (clock.getTime().isBefore(deadline)) {
            String healthyStatus = inspect(inspectHealthyStatusTimeout, "{{.State.Health.Status}}");
            if (isEmpty(healthyStatus)) {
                // [podman-compatibility] Если запустили контейнер через podman, статус надо брать по другому пути
                healthyStatus = inspect(inspectHealthyStatusTimeout, "{{.State.Healthcheck.Status}}");
            }
            healthyStatus = healthyStatus.replace(System.lineSeparator(), "");
            logger.info("Container {} healthy status - {}, expected to be {}",
                    runner.getImage(), healthyStatus, HEALTHY_STATUS);
            if (HEALTHY_STATUS.equals(healthyStatus)) {
                return;
            }
            try {
                clock.sleep(sleepBetweenHealthyCheck);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(e);
            }
        }
        throw new RuntimeTimeoutException(String.format("Docker container %s expected to become healthy in %d seconds",
                getContainerId(), waitContainerHealthyDuration.getSeconds()));
    }
}
