package ru.yandex.direct.process;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermissions;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.io.IOUtils;

import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.io.TempDirectory;

/**
 * Создание Docker-контейнера, в котором процесс будет запускаться от пользователя с таким же UID и GID, как и на хосте.
 * Если контейнер запускается на Linux-хосте и монтирует volumes, то пользователь на хосте будет иметь полный доступ к
 * создаваемым в контейнере файлам.
 */
@ParametersAreNonnullByDefault
public class DockerHostUserEntryPoint {
    public static final boolean IS_MACOS = System.getProperty("os.name").equals("Mac OS X");
    private static HostUserInfo hostUserInfo = null;
    private final TempDirectory tempDirectory;
    private String origEntryPoint = null;

    /**
     * @param tempDirectory  Временный каталог, в котором будет сгенерирован новый entrypoint-файл для контейнера.
     *                       Каталог должен существовать на момент старта контейнера.
     * @param origEntryPoint Исходный entrypoint, который указан в Docker-образе, если есть, или null
     */
    public DockerHostUserEntryPoint(TempDirectory tempDirectory, @Nullable String origEntryPoint)
            throws InterruptedException {
        this.tempDirectory = tempDirectory;
        this.origEntryPoint = origEntryPoint;
        synchronized (DockerHostUserEntryPoint.class) {
            if (hostUserInfo == null) {
                hostUserInfo = new HostUserInfo(System.getProperty("user.name"),
                        Long.valueOf(Processes.checkOutput("id", "-u").trim()),
                        Long.valueOf(Processes.checkOutput("id", "-g").trim()));
            }
        }
    }

    private Path generateEntrypoint() throws IOException {
        Path entrypointFile = Files.createTempFile(tempDirectory.getPath(), "host_user_entrypoint.", ".sh",
                PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwxr-xr-x")));
        try (InputStream sourceStream = DockerRunner.class.getResourceAsStream("host_user_entrypoint.sh");
             OutputStream output = Files.newOutputStream(entrypointFile)) {
            IOUtils.copy(sourceStream, output);
        }
        return entrypointFile;
    }

    /**
     * Добавить параметры в {@literal DockerRunner}, чтобы при запуске в нём был создан пользователь, идентичный
     * хостовому.
     */
    public DockerRunner apply(DockerRunner dockerRunner) {
        DockerRunner runner = dockerRunner;
        try {
            Path entrypointPath = generateEntrypoint();
            String containerEntrypoint = "/host_user_entrypoint.sh";
            HostUserInfo info = hostUserInfo;
            runner = runner
                    .withVolume(entrypointPath, Paths.get(containerEntrypoint), true)
                    .withEntrypoint(containerEntrypoint)
                    .withEnvironment("HOST_USERNAME", info.userName)
                    .withUser("root");
            if (origEntryPoint != null) {
                runner = runner
                        .withEnvironment("ORIGINAL_ENTRYPOINT", origEntryPoint);
            }
            if (IS_MACOS) {
                // Окружение такое же, как в docker-machine
                // На самом деле Docker for Mac хитрый. Всё, что пишется в docker-контейнере в volume на хост приезжает от имени
                // пользователя, под которым запущен Docker for Mac. То есть под macos можно не использовать DockerHostUserEntrypoint,
                // но пусть лучше используется: желательно, чтобы на разных операционных системах приложение работало по одному алгоритму.
                runner = runner
                        .withEnvironment("HOST_UID", "1000")
                        .withEnvironment("HOST_GID", "50");
            } else {
                runner = runner
                        .withEnvironment("HOST_UID", Long.toString(info.uid))
                        .withEnvironment("HOST_GID", Long.toString(info.gid));
            }
            return runner;
        } catch (IOException e) {
            throw new Checked.CheckedException(e);
        }
    }

    private static class HostUserInfo {
        final String userName;
        final long uid;
        final long gid;

        HostUserInfo(String userName, long uid, long gid) {
            this.userName = userName;
            this.uid = uid;
            this.gid = gid;
        }
    }
}
