package ru.yandex.direct.process;

import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

public class DockerRunner {
    public static final List<String> DOCKER_ARGS = Collections.unmodifiableList(Collections.emptyList());
    public static final List<String> DOCKER_RUN_ARGS = Collections.unmodifiableList(Collections.singletonList(
            "--detach"
    ));

    private final String image;
    private List<String> cmd;
    private final List<String> args;
    private final List<String> runArgs;
    private final Docker docker;
    private String hostname;

    public DockerRunner(Docker docker, String image) {
        this(docker, image, null, Collections.emptyList(), DOCKER_ARGS, DOCKER_RUN_ARGS);
    }

    private DockerRunner(Docker docker, String image, String hostname,
                         List<String> cmd, List<String> args, List<String> runArgs
    ) {
        this.image = image;
        this.cmd = new ArrayList<>(cmd);
        this.args = new ArrayList<>(args);
        this.runArgs = new ArrayList<>(runArgs);
        this.docker = docker;
        this.hostname = hostname;
    }

    public String getImage() {
        return image;
    }

    public Docker getDocker() {
        return docker;
    }

    public Optional<String> getHostname() {
        return Optional.ofNullable(hostname);
    }

    public DockerRunner copy() {
        return new DockerRunner(
                docker,
                image,
                hostname,
                new ArrayList<>(cmd),
                new ArrayList<>(args),
                new ArrayList<>(runArgs)
        );
    }

    public DockerRunner withRunArgs(String... additionalRunArgs) {
        this.runArgs.addAll(Arrays.asList(additionalRunArgs));
        return this;
    }

    public DockerRunner withCmd(String... cmd) {
        return withCmd(Arrays.asList(cmd));
    }

    public DockerRunner withCmd(List<String> cmd) {
        if (!this.cmd.isEmpty()) {
            throw new IllegalStateException(
                    "cmd is already set. Current value: " + this.cmd + ", attempted value: " + Arrays.asList(cmd)
            );
        }
        this.cmd = cmd;
        return this;
    }

    public DockerRunner withName(String name) {
        this.runArgs.add("--name=" + name);
        return this;
    }

    public DockerRunner withAutoRemove() {
        this.runArgs.add("--rm");
        return this;
    }

    public DockerRunner withHostname(String hostname) {
        this.runArgs.add("--hostname=" + hostname);
        this.hostname = hostname;
        return this;
    }

    public DockerRunner withNetwork(DockerNetwork network) {
        // В docker 1.13 аргументы --net и --net-alias были переименованы в --network и --network-alias
        // При этом старые названия параметров так же поддерживаются, хотя в docker run --help не выдаются.
        // Работает как минимум в docker 1.10 и 17.06
        this.runArgs.add("--net=" + network.getName());
        return this;
    }

    public DockerRunner withNetworkAlias(String alias) {
        this.runArgs.add("--net-alias=" + alias);
        return this;
    }

    public DockerRunner withEnvironment(String varName, String value) {
        this.runArgs.add("--env=" + varName + "=" + value);
        return this;
    }

    public DockerRunner withRestartAlways() {
        this.runArgs.add("--restart=always");
        return this;
    }

    public DockerRunner withVolume(Path source, Path target, boolean readOnly) {
        if (!target.isAbsolute()) {
            throw new IllegalArgumentException("target path must be absolute: " + target);
        }
        this.runArgs.add("--volume=" + source.normalize().toAbsolutePath() + ":" + target + (readOnly ? ":ro" : ""));
        return this;
    }

    public DockerRunner withPublishedPort(int port) {
        this.runArgs.add("--publish=" + Integer.toString(port));
        return this;
    }

    public DockerRunner withPublishedPort(int hostPort, int containerPort) {
        this.runArgs.add(String.format("--publish=%d:%d", hostPort, containerPort));
        return this;
    }

    public DockerRunner withEntrypoint(String cmd) {
        this.runArgs.add("--entrypoint=" + cmd);
        return this;
    }

    public DockerRunner withUser(String user) {
        this.runArgs.add("--user=" + user);
        return this;
    }

    public List<String> getCommandLine() {
        List<String> command = new ArrayList<>();
        command.addAll(args);
        command.add("run");
        command.addAll(runArgs);
        command.add(image);
        command.addAll(cmd);
        return command;
    }

    public DockerContainerId run() throws InterruptedException {
        return new DockerContainerId(docker.checkOutput(getCommandLine().toArray(new String[0])).trim());
    }

    public DockerContainerId run(Duration timeout) {
        return new DockerContainerId(docker.checkOutputUninterruptibly(
                timeout,
                getCommandLine().toArray(new String[0])).trim()
        );
    }
}
