package ru.yandex.direct.process;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import org.apache.commons.io.IOUtils;

import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.Transient;

import static java.util.Objects.requireNonNull;

/**
 * Запускает внешний процесс и позволяет читать его stdout, stderr и писать ему в stdin.
 * Старается делать это максимально безопасно в смысле дедлоков - не разрешает "висящие" пайпы.
 * <p>
 * Будьте внимательны, если дочерний процесс форкнет внука, который делает, скажем, sleep 1000000, то
 * читающие треды никогда не остановятся, даже после того как вы убъете дочерний процесс и закроете пайпы
 * со своей стороны. Это такая особенность линукса, если в одном треде у вас работает read(pipe_fd) (тред
 * заблокирован на syscall), то close(pipe_fd) в другом треде не поможет разблокировать его.
 * <p>
 * Теоретически, можно изловчиться и сделать так, чтобы читающие/пишущие треды не блокировались навечно,
 * но это только одна проблема. Вторая проблема заключается в том, что процесс-внук не будет убит и эту проблему
 * на стороне джавы решить нельзя.
 * <p>
 * Поэтому, если ты запускаешь дочерние процессы, которые потенциально могут запускать еще процессы, следует
 * использовать внешние утилиты, создающие отдельный pid namespace, тогда при убийстве дочернего процесса будут
 * убиты и все процессы запущенные им.
 * <p>
 * Примеры утилит:
 * vagga - https://github.com/tailhook/vagga (или docker на худой конец)
 * доморощенный chinit - https://a.yandex-team.ru/arc/trunk/arcadia/ads/yacontext/packages/yacontext-utils
 */
public class ProcessCommunicator implements AutoCloseable {
    public static class Builder {
        private static final AtomicInteger counter = new AtomicInteger(0);

        private final ProcessBuilder processBuilder;
        private String threadNamePrefix;
        private Duration gracefulStopTimeout;
        private Duration ioThreadsTimeout;
        private Consumer<OutputStream> stdinWriter;
        private Consumer<InputStream> stdoutReader;
        private Consumer<InputStream> stderrReader;
        private boolean allowStdinPipe;
        private boolean allowStdoutPipe;
        private boolean allowStderrPipe;
        private boolean closeStdin;
        private boolean raiseExceptionOnNonZeroExitCode;

        public Builder(ProcessBuilder processBuilder) {
            this.processBuilder = processBuilder;
            this.threadNamePrefix = "ProcessCommunicator" + counter.incrementAndGet();
            this.gracefulStopTimeout = Duration.ZERO;
            this.ioThreadsTimeout = Duration.ofSeconds(10);
            this.stdinWriter = null;
            this.stdoutReader = null;
            this.stderrReader = null;
            this.allowStdinPipe = true;
            this.allowStdoutPipe = false;
            this.allowStderrPipe = false;
            this.closeStdin = true;
            this.raiseExceptionOnNonZeroExitCode = true;
        }

        public Builder withThreadNamePrefix(String threadNamePrefix) {
            this.threadNamePrefix = threadNamePrefix;
            return this;
        }

        public Builder withGracefulStopTimeout(Duration gracefulStopTimeout) {
            this.gracefulStopTimeout = gracefulStopTimeout;
            return this;
        }

        public Builder withAllowInheritedStdin() {
            if (!processBuilder.redirectInput().equals(ProcessBuilder.Redirect.INHERIT)) {
                throw new IllegalStateException("Process stdin is not inherited");
            }
            this.allowStdinPipe = false;
            this.stdinWriter = null;
            this.closeStdin = false;
            return this;
        }

        public Builder withStdinWriter(Consumer<OutputStream> stdinWriter) {
            this.stdinWriter = requireNonNull(stdinWriter);
            this.allowStdinPipe = true;
            this.closeStdin = false;
            return this;
        }

        public Builder withStdoutReader(Consumer<InputStream> stdoutReader) {
            this.stdoutReader = requireNonNull(stdoutReader);
            this.allowStdoutPipe = true;
            return this;
        }

        public Builder withStderrReader(Consumer<InputStream> stderrReader) {
            this.stderrReader = requireNonNull(stderrReader);
            this.allowStderrPipe = true;
            return this;
        }

        public Builder withRaiseExceptionOnNonZeroExitCode(boolean raiseExceptionOnNonZeroExitCode) {
            this.raiseExceptionOnNonZeroExitCode = raiseExceptionOnNonZeroExitCode;
            return this;
        }

        public ProcessCommunicator build() {
            if (!allowStdinPipe && processBuilder.redirectInput().equals(ProcessBuilder.Redirect.PIPE)) {
                throw new IllegalStateException("stdin must not be redirected to pipe");
            }

            if (!allowStdoutPipe && processBuilder.redirectOutput().equals(ProcessBuilder.Redirect.PIPE)) {
                throw new IllegalStateException("stdout must not be redirected to pipe");
            }

            if (!allowStderrPipe && processBuilder.redirectError().equals(ProcessBuilder.Redirect.PIPE)) {
                throw new IllegalStateException("stderr must not be redirected to pipe");
            }

            try (Transient<ProcessCommunicator> communicatorHolder = new Transient<>(new ProcessCommunicator(
                    processBuilder,
                    threadNamePrefix,
                    gracefulStopTimeout,
                    ioThreadsTimeout,
                    stdinWriter,
                    stdoutReader,
                    stderrReader,
                    raiseExceptionOnNonZeroExitCode
            ))) {
                if (closeStdin) {
                    try {
                        communicatorHolder.item.processHolder.getProcess().getOutputStream().close();
                    } catch (IOException exc) {
                        throw new Checked.CheckedException(exc);
                    }
                }
                return communicatorHolder.pop();
            }
        }
    }

    private final ProcessHolder processHolder;
    private final Completer completer;
    private final Future<Void> waitFuture;
    private final Duration ioThreadsTimeout;

    // Принимает ProcessBuilder, а не ProcessHolder, потому что нужен контроль над процессом.
    // Чтобы остановить читающие треды, сначала нужно остановить сам процесс, а этого нельзя
    // сделать если время жизни ProcessHolder контроллируется вовне.
    private ProcessCommunicator(
            ProcessBuilder processBuilder,
            String threadNamePrefix,
            Duration gracefulStopTimeout,
            Duration ioThreadsTimeout,
            Consumer<OutputStream> stdinWriter,
            Consumer<InputStream> stdoutReader,
            Consumer<InputStream> stderrReader,
            boolean raiseExceptionOnNonZeroExitCode
    ) {
        this.ioThreadsTimeout = ioThreadsTimeout;
        try (Transient<ProcessHolder> transientProcessHolder = new Transient<>(
                new ProcessHolder(processBuilder, gracefulStopTimeout, raiseExceptionOnNonZeroExitCode)
        )) {
            // В замыкания нужно отдавать только процесс, а не transientProcessHolder
            Process process = transientProcessHolder.item.getProcess();

            Completer.Builder completerBuilder = new Completer.Builder(Duration.ofSeconds(0));

            waitFuture = completerBuilder.submitVoid(
                    threadNamePrefix + "-wait",
                    transientProcessHolder.item::waitFor
            );

            if (stdinWriter != null) {
                if (!processBuilder.redirectInput().equals(ProcessBuilder.Redirect.PIPE)) {
                    throw new IllegalArgumentException("stdin must be redirected to PIPE");
                }
                completerBuilder.submitVoid(threadNamePrefix + "-stdin", () -> {
                    // При завершении консьюмера закрываем пайп, во избежание дедлока
                    try (OutputStream stdin = process.getOutputStream()) {
                        stdinWriter.accept(stdin);
                    }
                });
            }

            if (stdoutReader != null) {
                if (!processBuilder.redirectOutput().equals(ProcessBuilder.Redirect.PIPE)) {
                    throw new IllegalArgumentException("stdout must be redirected to PIPE");
                }
                completerBuilder.submitVoid(threadNamePrefix + "-stdout", () -> {
                    // При завершении консьюмера закрываем пайп, во избежание дедлока
                    try (InputStream stdout = process.getInputStream()) {
                        stdoutReader.accept(stdout);
                    }
                });
            }

            if (stderrReader != null) {
                if (!processBuilder.redirectError().equals(ProcessBuilder.Redirect.PIPE)) {
                    throw new IllegalArgumentException("stderr must be redirected to PIPE");
                }
                completerBuilder.submitVoid(threadNamePrefix + "-stderr", () -> {
                    // При завершении консьюмера закрываем пайп, во избежание дедлока
                    try (InputStream stderr = process.getErrorStream()) {
                        stderrReader.accept(stderr);
                    }
                });
            }

            this.processHolder = transientProcessHolder.pop();
            this.completer = completerBuilder.build();
        }
    }

    /**
     * выполнить, и кинуть исключение, если код возврата не 0
     */
    public static void communicateSafe(
            ProcessBuilder processBuilder,
            Consumer<OutputStream> stdinWriter,
            Consumer<InputStream> stdoutReader,
            Consumer<InputStream> stderrReader
    )
            throws InterruptedException {
        communicate(processBuilder, stdinWriter, stdoutReader, stderrReader, true);
    }

    /**
     * выполнить, и даже если код возврата не 0, то исключения не будет
     */
    public static void communicateUnsafe(
            ProcessBuilder processBuilder,
            Consumer<OutputStream> stdinWriter,
            Consumer<InputStream> stdoutReader,
            Consumer<InputStream> stderrReader
    )
            throws InterruptedException {
        communicate(processBuilder, stdinWriter, stdoutReader, stderrReader, false);
    }

    private static void communicate(
            ProcessBuilder processBuilder,
            Consumer<OutputStream> stdinWriter,
            Consumer<InputStream> stdoutReader,
            Consumer<InputStream> stderrReader,
            boolean throwException
    )
            throws InterruptedException {
        try (
                ProcessCommunicator communicator = new Builder(processBuilder)
                        .withStdinWriter(stdinWriter)
                        .withStdoutReader(stdoutReader)
                        .withStderrReader(stderrReader)
                        .withRaiseExceptionOnNonZeroExitCode(throwException)
                        .build()
        ) {
            communicator.waitFor();
        }
    }

    public void waitFor() throws InterruptedException {
        try {
            completer.waitAll();
        } catch (Completer.CompleterException exc) {
            checkIfProcessFailed();
            throw exc;
        }
    }

    public boolean waitFor(Duration timeout) throws InterruptedException {
        try {
            return completer.waitAll(timeout);
        } catch (Completer.CompleterException exc) {
            checkIfProcessFailed();
            throw exc;
        }
    }

    private void checkIfProcessFailed() {
        if (waitFuture.isDone()) {
            try {
                Interrupts.resurrectingRun(waitFuture::get);
            } catch (ExecutionException exc) {
                // Если процесс завершился с ошибкой, нужно бросить оригинальное исключение из ProcessHolder.waitFor()
                if (exc.getCause() instanceof ProcessExitStatusException) {
                    throw (ProcessExitStatusException) exc.getCause();
                }
                if (exc.getCause() instanceof ProcessForciblyDestroyedException) {
                    throw (ProcessForciblyDestroyedException) exc.getCause();
                }
                throw new IllegalStateException("Got unexpected exception while waiting for process to exit", exc);
            }
        }
    }

    @Override
    public void close() {
        try (Completer ignored = completer) {
            processHolder.close();
            Interrupts.waitUninterruptibly(ioThreadsTimeout, completer::waitAll);
        }
    }

    public static class StringReader implements Consumer<InputStream> {
        private volatile String content;
        private final Charset charset;

        public StringReader(Charset charset) {
            this.charset = charset;
            this.content = null;
        }

        public String getContent() {
            if (content == null) {
                throw new IllegalStateException("Content has not been read yet");
            }
            return content;
        }

        @Override
        public void accept(InputStream inputStream) {
            try {
                content = IOUtils.toString(inputStream, charset.name());
            } catch (IOException exc) {
                throw new Checked.CheckedException(exc);
            }
        }
    }

    public static class LineReader implements Consumer<InputStream> {
        private final Charset charset;
        private final Consumer<String> lineConsumer;

        public LineReader(Charset charset, Consumer<String> lineConsumer) {
            this.charset = charset;
            this.lineConsumer = lineConsumer;
        }

        @Override
        public void accept(InputStream inputStream) {
            try (
                    InputStreamReader reader = new InputStreamReader(inputStream, charset);
                    BufferedReader buffered = new BufferedReader(reader)
            ) {
                buffered.lines().forEachOrdered(lineConsumer);
            } catch (IOException exc) {
                throw new Checked.CheckedException(exc);
            }
        }
    }
}
