package ru.yandex.direct.process;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Interrupts;

public class ProcessHolder implements AutoCloseable {
    private final Process process;
    private final List<String> command;
    private final Duration gracefulStopTimeout;
    private final boolean raiseExceptionOnNonZeroExitCode;
    private boolean forciblyDestroyed;

    public ProcessHolder(ProcessBuilder processBuilder) {
        this(processBuilder, Duration.ZERO, true);
    }

    public ProcessHolder(ProcessBuilder processBuilder, Duration gracefulStopTimeout) {
        this(processBuilder, gracefulStopTimeout, true);
    }

    public ProcessHolder(ProcessBuilder processBuilder, Duration gracefulStopTimeout,
                         boolean raiseExceptionOnNonZeroExitCode) {
        this.forciblyDestroyed = false;
        this.gracefulStopTimeout = gracefulStopTimeout;
        this.command = new ArrayList<>(processBuilder.command());
        this.raiseExceptionOnNonZeroExitCode = raiseExceptionOnNonZeroExitCode;
        try {
            this.process = processBuilder.start();
        } catch (IOException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    /**
     * Этот конструктор исключительно для тестов
     */
    ProcessHolder(Process process, Duration gracefulStopTimeout) {
        this.forciblyDestroyed = false;
        this.gracefulStopTimeout = gracefulStopTimeout;
        this.raiseExceptionOnNonZeroExitCode = true;
        this.command = new ArrayList<>();
        this.process = process;
    }

    public Process getProcess() {
        return process;
    }

    /**
     * Вызывает {@code Process.destroy()} и ждет завершения.
     * Если процесс не завершился за timeout, вызывает {@code Process.destroyForcibly()}.
     * <p>
     * Класс должен быть потокобезопасным. Сделать такую синхронизацию, чтобы два параллельных stop(Duration)
     * с разными таймаутами работали корректно, я не осилил, поэтому метод приватный.
     *
     * @throws ProcessExitStatusException        {@see waitFor()}.
     * @throws ProcessForciblyDestroyedException {@see waitFor()}.
     */
    private synchronized void stop(Duration timeout) {
        if (isStopped()) {
            reportNonZeroExitCode();
        } else {
            if (needsGracefulStop()) {
                process.destroy();
                if (!Interrupts.waitUninterruptibly(timeout, this::waitFor)) {
                    process.destroyForcibly();
                    forciblyDestroyed = true;
                }
            } else {
                process.destroyForcibly();
                forciblyDestroyed = true;
            }
            Interrupts.resurrectingRun(this::waitFor);
        }
    }

    /**
     * Вызывает {@code Process.destroy()} и ждет завершения.
     * Если процесс не завершился за gracefulStopTimeout, вызывает {@code Process.destroyForcibly()}.
     *
     * @throws ProcessExitStatusException        {@see waitFor()}.
     * @throws ProcessForciblyDestroyedException {@see waitFor()}.
     */
    public void stop() {
        stop(gracefulStopTimeout);
    }

    /**
     * Ждет завершения процесса, но не дольше timeout.
     *
     * @return {@code true} если процесс завершился с нулевым кодом возврата за отведенный timeout,
     * {@code false} если процесс не завершился за отведенный timeout,
     * если процесс завершился с ненулевым кодом возврата - случается ProcessExitStatusException.
     * @throws InterruptedException              {@see waitFor()}.
     * @throws ProcessExitStatusException        {@see waitFor()}.
     * @throws ProcessForciblyDestroyedException {@see waitFor()}.
     */
    public boolean waitFor(Duration timeout) throws InterruptedException {
        boolean exited = process.waitFor(timeout.toMillis(), TimeUnit.MILLISECONDS);
        if (exited) {
            reportNonZeroExitCode();
        }
        return exited;
    }

    /**
     * Ждет завершения процесса.
     *
     * @throws InterruptedException              если ожидание было прервано
     * @throws ProcessExitStatusException        если процесс завершился с ненулевым кодом возврата
     *                                           и параметр {@link #raiseExceptionOnNonZeroExitCode} истинен
     * @throws ProcessForciblyDestroyedException если процесс был завершен через {@code Process.destroyForcibly()}.
     *                                           Однако, если для процесса был указан нулевой gracefulStopTimeout (т.е. он всегда будет убиваться
     *                                           через destroyForcibly), то исключение брошено не будет.
     */
    public void waitFor() throws InterruptedException {
        process.waitFor();
        reportNonZeroExitCode();
    }

    /**
     * Останавливает процесс и бросает исключение в случае ненормальной остановки.
     * Однако, если такое исключение уже было брошено из stop или waitFor, то повторно оно брошено не будет.
     *
     * @throws ProcessExitStatusException        если процесс завершился с ненулевым кодом возврата и параметр
     *                                           {@link #raiseExceptionOnNonZeroExitCode} истинен.
     *                                           При этом, если ранее это исключение уже было брошено,
     *                                           то повторно в close оно брошено не будет.
     * @throws ProcessForciblyDestroyedException если процесс был уничтожен через destroyForcibly
     */
    @Override
    public synchronized void close() {
        // не бросаем ProcessExitStatusException и ProcessForciblyDestroyedException повторно.
        if (!isStopped()) {
            stop();
        }
    }

    private synchronized void reportNonZeroExitCode() {
        if (forciblyDestroyed) {
            if (needsGracefulStop()) {
                throw new ProcessForciblyDestroyedException(command);
            }
        } else {
            int exitValue = process.exitValue();
            if (exitValue != 0 && raiseExceptionOnNonZeroExitCode) {
                throw new ProcessExitStatusException(command, exitValue);
            }
        }
    }

    private boolean needsGracefulStop() {
        return !gracefulStopTimeout.isZero() && !gracefulStopTimeout.isNegative();
    }

    private synchronized boolean isStopped() {
        return !process.isAlive() || forciblyDestroyed;
    }
}
