package ru.yandex.direct.process;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.apache.commons.io.IOUtils;

import ru.yandex.direct.utils.Checked;

public class Processes {
    private static final Path DEV_NULL = Paths.get("/dev/null");
    private static final ProcessBuilder.Redirect DEV_NULL_REDIRECT = ProcessBuilder.Redirect.to(DEV_NULL.toFile());
    private static final Charset CHARSET = StandardCharsets.UTF_8;

    private Processes() {
        throw new IllegalAccessError("Utility class");
    }

    public static void checkCall(String... command) throws InterruptedException {
        checkCall(
                new ProcessBuilder(command)
                        .redirectInput(ProcessBuilder.Redirect.INHERIT)
                        .redirectOutput(ProcessBuilder.Redirect.INHERIT)
                        .redirectError(ProcessBuilder.Redirect.INHERIT)
        );
    }

    public static void checkCallSilent(String... command) throws InterruptedException {
        checkCall(
                new ProcessBuilder(command)
                        .redirectInput(ProcessBuilder.Redirect.INHERIT)
                        .redirectOutput(DEV_NULL_REDIRECT)
                        .redirectError(DEV_NULL_REDIRECT)
        );
    }

    public static void checkCall(ProcessBuilder processBuilder) throws InterruptedException {
        try (ProcessHolder process = new ProcessHolder(processBuilder)) {
            process.waitFor();
        }
    }

    public static String checkOutput(String... command) throws InterruptedException {
        return checkOutput(new ProcessBuilder(command));
    }

    public static String checkOutput(ProcessBuilder processBuilder) throws InterruptedException {
        ProcessCommunicator.StringReader stdoutReader = new ProcessCommunicator.StringReader(CHARSET);

        try (ProcessCommunicator communicator =
                     new ProcessCommunicator.Builder(
                             processBuilder
                                     .redirectInput(ProcessBuilder.Redirect.PIPE)
                                     .redirectError(ProcessBuilder.Redirect.INHERIT)
                                     .redirectOutput(ProcessBuilder.Redirect.PIPE)
                     )
                             .withStdoutReader(stdoutReader).build()) {
            communicator.waitFor();
        }

        return stdoutReader.getContent();
    }

    /**
     * Выполнить процесс. Исключение не кидается не смотря на ненулеквой код возврата
     */
    public static CommunicationResult communicateUnsafe(ProcessBuilder processBuilder, String input)
            throws InterruptedException {
        ProcessCommunicator.StringReader stdoutReader = new ProcessCommunicator.StringReader(CHARSET);
        ProcessCommunicator.StringReader stderrReader = new ProcessCommunicator.StringReader(CHARSET);

        ProcessCommunicator.communicateUnsafe(
                processBuilder,
                stdin -> Checked.<IOException>run(() -> IOUtils.write(input, stdin, CHARSET.name())),
                stdoutReader,
                stderrReader
        );

        return new CommunicationResult(stdoutReader.getContent(), stderrReader.getContent());
    }

    public static CommunicationResult communicateUnsafe(ProcessBuilder processBuilder) throws InterruptedException {
        return communicateUnsafe(processBuilder, "");
    }

    /**
     * Выполнить процесс. Если код возврата не 0, получим исключение
     */
    public static CommunicationResult communicateSafe(ProcessBuilder processBuilder, String input)
            throws InterruptedException {
        ProcessCommunicator.StringReader stdoutReader = new ProcessCommunicator.StringReader(CHARSET);
        ProcessCommunicator.StringReader stderrReader = new ProcessCommunicator.StringReader(CHARSET);

        try {
            ProcessCommunicator.communicateSafe(
                    processBuilder,
                    stdin -> Checked.<IOException>run(() -> IOUtils.write(input, stdin, CHARSET.name())),
                    stdoutReader,
                    stderrReader
            );
        } catch (InterruptedException e) {
            throw e;
        } catch (RuntimeException e) {
            e.addSuppressed(new RuntimeException(
                    "stdout: \"" + stdoutReader.getContent() + "\", stderr: \"" + stderrReader.getContent() + "\""
            ));
            throw e;
        }

        return new CommunicationResult(stdoutReader.getContent(), stderrReader.getContent());
    }

    public static CommunicationResult communicateSafe(ProcessBuilder processBuilder) throws InterruptedException {
        return communicateSafe(processBuilder, "");
    }

    @SuppressWarnings("squid:S1166") // не нужно перевыбрасывать пойманное исключение
    public static boolean commandIsExecutable(String... args) throws InterruptedException {
        try {
            Processes.checkCallSilent(args);
            return true;
        } catch (RuntimeException exc) {
            return false;
        }
    }

    @SuppressWarnings({"squid:S1147", "squid:S106"}) // осознанный вывод в System.err и вызов System.exit()
    public static void exitWithFailure(String message) {
        System.err.println(message);
        System.exit(1);
    }

    public static class CommunicationResult {
        private final String stdout;
        private final String stderr;

        public CommunicationResult(String stdout, String stderr) {
            this.stdout = stdout;
            this.stderr = stderr;
        }

        public String getStdout() {
            return stdout;
        }

        public String getStderr() {
            return stderr;
        }
    }
}
