# -*- coding: utf-8 -*-

import logging
import tempfile
import time
import threading
import requests
from BaseHTTPServer import BaseHTTPRequestHandler, HTTPServer
from path import Path

import yatest.common
from yatest.common import network

CHECKER_PATH = "direct/libs/python/alive-checker/alive-checker"
logger = logging.getLogger("test_logger")


class Test():
    @classmethod
    def setup_class(cls):
        cls.pm = network.PortManager()
        cls.port = cls.pm.get_port(30000)

        cls.httpd = HTTPServer(('127.0.0.1', cls.port), MockReqHandler)

        cls.thread = threading.Thread(target=cls.httpd.serve_forever)
        cls.thread.setdaemon = True
        cls.thread.start()

        cls.url = "http://127.0.0.1:{}/alive".format(cls.port)
        cls.file = Path(tempfile.mkstemp()[1])
        cls.file.abspath()

    @classmethod
    def teardown_class(cls):
        cls.httpd.shutdown()
        cls.pm.release()
        if cls.file.isfile():
            cls.file.unlink()

    def setup(self):
        MockReqHandler.reset()
        if self.file.isfile():
            self.file.unlink()

    @classmethod
    def run_checker(cls, once=True, monrun=False, timeout=None):
        cmd = [yatest.common.binary_path(CHECKER_PATH), "-u", cls.url, "-f", cls.file]
        logger.error(str(cmd))
        if once:
            cmd.append("--once")
        if monrun:
            cmd.append("--monrun")
        if timeout is not None:
            cmd.extend(["--timeout", str(timeout)])
        yatest.common.execute(cmd)

    def test_http_server_works(self):
        MockReqHandler.reset()
        resp = requests.get(self.url)
        assert resp.content == "ok"

    def test_file(self):
        MockReqHandler.reset()
        self.run_checker()
        assert self.file.exists()
        assert self.file.text() == 'ok'

    def test_file_then_error(self):
        MockReqHandler.reset()
        self.run_checker()
        assert self.file.exists()
        assert self.file.text() == 'ok'

        MockReqHandler.code = 500
        self.run_checker()
        assert not self.file.exists()

    def test_file_then_timeout(self):
        MockReqHandler.reset()
        self.run_checker()
        assert self.file.isfile()
        assert self.file.text() == 'ok'

        MockReqHandler.sleep = 1
        start = time.time()
        self.run_checker(timeout=0.01)
        assert time.time() - start < 0.5
        assert not self.file.isfile()


class MockReqHandler(BaseHTTPRequestHandler):
    @classmethod
    def reset(cls):
        cls.code = 200
        cls.resp = "ok"
        cls.sleep = None

    def _set_headers(self, status):
        self.send_response(status)
        self.send_header('Content-type', 'text/html')
        self.end_headers()

    def do_GET(self):
        if self.sleep is not None:
            time.sleep(self.sleep)
        logger.error(self.path)
        if self.path == '/alive':
            self._set_headers(self.code)
            self.wfile.write(self.resp)
        else:
            self._set_headers(404)
            self.wfile.write("not found")
