import re
import sys
import json
from gzip import GzipFile
from StringIO import StringIO
from base64 import b64decode

import click


JSON_START = re.compile(r'[\{\{]')


@click.command('json', help="format jsons parts, embedded in line", context_settings=dict(help_option_names=['-h', '--help']))
@click.option('--min-json-length', '-s', default=20, type=int,
              help="minimum length of json to process")
@click.option('--unescape/--no-unescape', '-u', default=False,
              help="unescape resulting json, suitable for json in json")
@click.option('--compressed', '-c', default=False, is_flag=True,
              help="every line is gziped and base64-ed")
def cli(min_json_length, unescape, compressed):
    decoder = json.JSONDecoder()
    for line in sys.stdin:
        line = line.strip()
        if compressed:
            line = GzipFile(fileobj=StringIO(b64decode(line))).read()

        printed = 0
        cur = 0
        while cur < len(line):
            if line[cur] == '{' or line[cur] == '[':
                try:
                    obj, end = decoder.raw_decode(line, cur)
                    if end - cur >= min_json_length:
                        sys.stdout.write(line[printed:cur])
                        sys.stdout.write(_json_fmt(obj, unescape))
                        printed = end
                        cur = end - 1
                except ValueError:
                    pass
            cur += 1

        sys.stdout.write(line[printed:cur])
        sys.stdout.write("\n")


def _json_fmt(obj, unescape):
    ret = json.dumps(obj, sort_keys=True, indent=4, ensure_ascii=False)
    if unescape:
        try:
            return ret.decode('unicode_escape')
        except UnicodeDecodeError:
            return ret
    else:
        return ret
