# -*- coding: utf-8 -*-

import ast
import operator


class SimpleEval(object):
    """
    Обработка простейших выражений:
    expr = ast.parse("a+1")
    obj = namedtuple('X', ['x'])(3)
    print SimpleEval(expr, obj).eval()
    # выведет 3
    """
    operators = {
        ast.Add: operator.add,
        ast.Sub: operator.sub,
        ast.Mult: operator.mul,
        ast.Div: operator.truediv,
        ast.Mod: operator.mod,
        ast.USub: operator.neg,

        ast.Eq: operator.eq,
        ast.NotEq: operator.ne,
        ast.Lt: operator.lt,
        ast.LtE: operator.le,
        ast.Gt: operator.gt,
        ast.GtE: operator.ge,

        ast.Not: operator.not_,

        ast.Or: operator.or_,
        ast.And: operator.and_,
        }

    def __init__(self, expr, data):
        self.expr = expr
        self.data = data

    def eval(self):
        return self._eval(self.expr.body)

    def _eval(self, node):
        if isinstance(node, ast.Num):
            return node.n
        elif isinstance(node, ast.Str):
            return node.s
        elif isinstance(node, ast.Name):
            return getattr(self.data, node.id)
        elif isinstance(node, ast.Attribute):
            return getattr(self._eval(node.value), node.attr)
        elif isinstance(node, ast.Call):
            return self._eval(node.func)(*[self._eval(a) for a in node.args])
        elif isinstance(node, ast.Compare):
            left = self._eval(node.left)
            for comparison_op, right_expr in zip(node.ops, node.comparators):
                right = self._eval(right_expr)
                if not self._op(comparison_op)(left, right):
                    return False
                left = right
            return True
        elif isinstance(node, ast.UnaryOp):
            return self._op(node.op)(self._eval(node.operand))
        elif isinstance(node, ast.BinOp):
            return self._op(node.op)(self._eval(node.left), self._eval(node.right))
        elif isinstance(node, ast.BoolOp):
            it = iter(node.values)
            ret = self._eval(it.next())
            for v in it:
                ret = self._op(node.op)(ret, self._eval(v))
            return ret
        else:
            raise Exception("Unsupported node: " + str(node))

    def _op(self, op):
        if type(op) not in self.operators:
            raise Exception("Unsupported operator: " + op)
        return self.operators[type(op)]
