import sys
from datetime import datetime
import re

import dateutil.parser
import click
from click import style


DATETIME_RE = re.compile(r'^(\S+?:)?(\d\d\d\d-\d\d-\d\d:\d\d:\d\d:\d\d)')
LOG_LEVEL_STYLES = {
    'FATAL': {'bg': 'red'},
    'ERROR': {'fg': 'red'},
    'WARN': {'fg': 'red'},
    'INFO': {'fg': 'blue'},
    'DEBUG': {'fg': 'green'},
    'TRACE': {'fg': 'green'},
}
LOG_LEVEL_RE = re.compile(' (' + '|'.join(LOG_LEVEL_STYLES.keys()) + ') ')
SKIP_NL_REPLACE_RE = re.compile('org.jooq.tools.LoggerListener')


def validate_datetime(ctx, param, value):
    if value is None:
        return None
    try:
        return dateutil.parser.parse(value)
    except ValueError:
        raise click.BadParameter("Incorrect datetime format, see dateutil.parser documentation")


@click.command('messages', help="messages.log filter and prettifier", context_settings=dict(help_option_names=['-h', '--help']))
@click.option('--from', '-f', 'from_datetime', default=None, callback=validate_datetime,
              help="from datetime")
@click.option('--to', '-t', 'to_datetime', default=None, callback=validate_datetime,
              help="to datetime")
@click.option('--lines-limit', '-n', default=None, type=int,
              help="count of lines of stacktraces to print")
@click.option('--pretty', default=True, is_flag=True,
              help="strip unuseless fields of messages log")
def cli(from_datetime, to_datetime, lines_limit, pretty):
    for line in sys.stdin:
        try:
            out = format_messages_line(line, from_datetime, to_datetime, lines_limit, pretty)
            if out is not None:
                click.echo(out)
        except Exception as e:
            click.echo(style("parse error", fg='red') + ": " + str(e))
            sys.stdout.write(line)


def format_messages_line(line, from_datetime, to_datetime, lines_limit, pretty):
    line = line.strip()
    dt_str, meta, data = line.split(' ', 2)

    dt_matcher = DATETIME_RE.match(dt_str)
    if dt_matcher is None:
        raise ValueError("Incorrect line format")

    dt = datetime.strptime(dt_matcher.group(2), "%Y-%m-%d:%H:%M:%S")
    if from_datetime is not None and dt < from_datetime \
       or to_datetime is not None and dt > to_datetime:
        return None

    if not SKIP_NL_REPLACE_RE.search(data):
        data_lines = data.split('|')
    else:
        data_lines = [data]

    if lines_limit is not None:
        data_lines = data_lines[0:lines_limit]
    if data_lines:
        data_lines[0] = LOG_LEVEL_RE.sub(lambda m: " " + style(m.group(1), **LOG_LEVEL_STYLES[m.group(1)]) + " ", data_lines[0])

    host, service_method, ids = meta.split(',')
    trace_id, parent_id, span_id = ids.split(':')

    if pretty:
        return "{} {},{} {}".format(dt_str, service_method, span_id, "\n    ".join(data_lines))
    else:
        return "{} {} {}".format(dt_str, meta, "\n    ".join(data_lines))
