import sys
import re

import dateutil.parser


import click


def validate_datetime(ctx, param, value):
    if value is None:
        return None
    try:
        return dateutil.parser.parse(value)
    except ValueError:
        raise click.BadParameter("Incorrect datetime format, see dateutil.parser documentation")


@click.command('thread-dumps-old', help="Old tool for filtering java thread-dumps logs (raw-text format)", context_settings=dict(help_option_names=['-h', '--help']))
@click.option('--from', '-f', 'from_datetime', default=None, callback=validate_datetime,
              help="from datetime")
@click.option('--to', '-t', 'to_datetime', default=None, callback=validate_datetime,
              help="to datetime")
@click.option('--filter', '-E', 'thread_filter', default=None,
              help="regexp for filtering threads by name")
@click.option('--count', '-n', default=None, type=int,
              help="count of dumps to print")
def cli(from_datetime, to_datetime, thread_filter, count):
    regexp = re.compile(r'^\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d\n')
    thread_filter_rx = re.compile('^"' + thread_filter + '".*') if thread_filter is not None else None

    print_dump = False
    print_thread = True
    found_count = 0
    for line in sys.stdin:
        if regexp.match(line):
            if count is not None and found_count >= count:
                break

            dt = dateutil.parser.parse(line)
            print_dump = (from_datetime is None or dt >= from_datetime) \
                         and (to_datetime is None or dt <= to_datetime)

            if print_dump:
                print_thread = True
                found_count += 1

        if thread_filter_rx and line.startswith('"'):
            print_thread = bool(thread_filter_rx.match(line))

        if print_dump and print_thread:
            sys.stdout.write(line)
