package ru.yandex.direct.queryrec;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.CharMatcher;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.queryrec.model.Alphabet;
import ru.yandex.direct.queryrec.model.Language;

import static java.util.function.Function.identity;

@ParametersAreNonnullByDefault
@Component
public class LanguageRecognizer {

    private final List<Language> supportedLanguages;
    private final List<Alphabet.Type> supportedAlphabetTypes;

    private final Map<Language, Map<Alphabet.Type, CharMatcher>> charMatchersByLanguage;

    public LanguageRecognizer() {
        supportedLanguages = Arrays.asList(Language.values());
        supportedAlphabetTypes = Arrays.asList(Alphabet.Type.values());

        charMatchersByLanguage = getCharMatchersByLanguage();
    }

    /**
     * Возвращает мап (язык -> буквы этого языка, не встречающиеся в других языках). Используются только языки,
     * поддерживаемые в Директе.
     *
     * @return мап (язык -> буквы этого языка, не встречающиеся в других языках)
     */
    private Map<Language, Map<Alphabet.Type, CharMatcher>> getCharMatchersByLanguage() {
        return StreamEx.of(supportedLanguages)
                .toMap(identity(), this::getCharMatchersForLanguage);
    }

    private Map<Alphabet.Type, CharMatcher> getCharMatchersForLanguage(Language language) {
        return StreamEx.of(supportedAlphabetTypes)
                .filter(language::hasAlphabetOfType)
                .mapToEntry(identity(), alphabetType -> new HashSet<>(language.getAllLettersByAlphabetType(alphabetType)))
                .peekKeyValue((alphabetType, letters) ->
                        StreamEx.of(supportedLanguages)
                                .filter(anotherLanguage ->
                                        anotherLanguage != language && anotherLanguage.hasAlphabetOfType(alphabetType))
                                .forEach(anotherLanguage ->
                                        letters.removeAll(anotherLanguage.getAllLettersByAlphabetType(alphabetType))))
                .mapValues(letters -> StreamEx.of(letters).joining())
                .mapValues(CharMatcher::anyOf)
                .toMap();
    }

    /**
     * Возвращает множество возможных кириллических языков для заданного текста.
     * Множество формируется на основе уникальных символов, не встречающихся в других языках.
     *
     * @param text  текст
     * @return множество возможных языков
     */
    Set<Language> findPossibleCyrillicLanguagesByUniqueLetters(String text) {
        return findPossibleLanguagesForAlphabetType(Alphabet.Type.CYRILLIC, text, Set.of());
    }

    /**
     * Возвращает множество возможных латинских языков для заданного текста.
     * Множество формируется на основе уникальных символов, не встречающихся в других языках:
     * если в тексте есть уникальные символы какого-либо языка — этот язык добавляется во множество.
     *
     * @param text  текст
     * @param recognizedLanguages — множество языков, которым мы ограничиваем ответ.
     *                              При пустом множестве ограничение не применяем.
     * @return множество возможных языков
     */
    Set<Language> findPossibleLatinLanguagesByUniqueLetters(String text, Set<String> recognizedLanguages) {
        return findPossibleLanguagesForAlphabetType(Alphabet.Type.LATIN, text, recognizedLanguages);
    }

    /**
     * Возвращает множество возможных языков заданного типа для текста.
     *
     * @param alphabetType        — тип алфавита
     * @param text                — текст
     * @param recognizedLanguages — множество языков, которым мы ограничиваем ответ.
     *                              При пустом множестве ограничение не применяем.
     * @return множество возможных языков заданного типа
     */
    private Set<Language> findPossibleLanguagesForAlphabetType(Alphabet.Type alphabetType,
            String text,
            Set<String> recognizedLanguages) {
        return EntryStream.of(charMatchersByLanguage)
                .filterKeys(language -> language.hasAlphabetOfType(alphabetType)
                        && (recognizedLanguages.isEmpty()
                            || recognizedLanguages.contains(language.getIso639Code().toString())))
                .mapValues(charMatcherByAlphabetType -> charMatcherByAlphabetType.get(alphabetType))
                .filterValues(charMatcher -> charMatcher.matchesAnyOf(text))
                .keys()
                .toSet();
    }
}
