package ru.yandex.direct.queryrec;

import java.io.IOException;
import java.io.InputStream;
import java.util.EnumMap;
import java.util.Map;

import javax.annotation.PreDestroy;

import com.google.common.io.ByteStreams;

import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

/**
 * У класса публичный только конструктор - чтобы объект можно было инициализировать единожды,
 * и на основе его создавать много QueryrecService
 */
public class QueryrecJni {
    private static final String DICT = "queryrec.dict";
    private static final String WEIGHTS = "queryrec.weights";
    private static final String FILTERS = "queryrec.filters";

    private volatile boolean initiliazed;
    private volatile boolean destroyed;
    private volatile long handle;

    static {
        System.loadLibrary("direct-queryrec-jni");
    }

    public QueryrecJni(boolean lazy) {
        if (!lazy) {
            init();
        }
    }

    private void initIfNeeded() {
        if (destroyed) {
            throw new IllegalStateException("Error on init queryrec: lib is already destroyed");
        }
        if (initiliazed) {
            return;
        }
        init();
    }

    private synchronized void init() {
        if (destroyed) {
            throw new IllegalStateException("Error on init queryrec: lib is already destroyed");
        }
        if (initiliazed) {
            return;
        }
        byte[] dictData, weightsData, regexpData;

        try (TraceProfile ignore = Trace.current().profile("queryrec:init");
             InputStream dictInput = QueryrecJni.class.getResourceAsStream(DICT);
             InputStream weightInput = QueryrecJni.class.getResourceAsStream(WEIGHTS);
             InputStream filtersInput = QueryrecJni.class.getResourceAsStream(FILTERS)) {
            dictData = ByteStreams.toByteArray(dictInput);
            weightsData = ByteStreams.toByteArray(weightInput);
            regexpData = ByteStreams.toByteArray(filtersInput);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        handle = create(dictData, weightsData, regexpData);
        initiliazed = true;
    }

    @PreDestroy
    synchronized void destroy() {
        if (!initiliazed || destroyed) {
            return;
        }
        destroy(handle);
        destroyed = true;
    }

    Map<Language, Double> recognize(String parse) {
        initIfNeeded();

        double[] array = recognize(parse, handle);
        return buildLangsToProbMap(array);
    }


    private Map<Language, Double> buildLangsToProbMap(double[] array) {
        Map<Language, Double> langsToProb = new EnumMap<>(Language.class);
        for (int i = 0; i < array.length; i = i + 2) {
            int langValue = (int) array[i];
            Language language = Language.getByValue(langValue);
            double prob = array[i + 1];

            langsToProb.put(language, prob);
        }

        return langsToProb;
    }


    private static native long create(byte[] dictData, byte[] weightsData, byte[] regexpData);

    private static native void destroy(long handle);

    private static native double[] recognize(String phrase, long handle);
}
