package ru.yandex.direct.queryrec;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.CharMatcher;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.queryrec.model.Alphabet;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Comparator.comparing;
import static ru.yandex.direct.queryrec.QueryrecHelper.ALL_CYRILLIC_LETTERS_EXCEPT_RUSSIAN_MATCHER;
import static ru.yandex.direct.queryrec.QueryrecHelper.ALL_CYRILLIC_LETTERS_MATCHER;
import static ru.yandex.direct.queryrec.QueryrecHelper.ALL_LATIN_LETTERS_MATCHER;
import static ru.yandex.direct.queryrec.QueryrecHelper.characterStream;
import static ru.yandex.direct.queryrec.QueryrecUtils.getBestAcceptableLanguage;
import static ru.yandex.direct.queryrec.QueryrecUtils.isCyrillicUzbek;
import static ru.yandex.direct.queryrec.QueryrecUtils.isLatinUzbek;
import static ru.yandex.direct.queryrec.model.Language.BELARUSIAN;
import static ru.yandex.direct.queryrec.model.Language.ENGLISH;
import static ru.yandex.direct.queryrec.model.Language.GERMAN;
import static ru.yandex.direct.queryrec.model.Language.KAZAKH;
import static ru.yandex.direct.queryrec.model.Language.RUSSIAN;
import static ru.yandex.direct.queryrec.model.Language.TURKISH;
import static ru.yandex.direct.queryrec.model.Language.UKRAINIAN;
import static ru.yandex.direct.queryrec.model.Language.UNKNOWN;
import static ru.yandex.direct.queryrec.model.Language.UZBEK;
import static ru.yandex.direct.queryrec.model.Language.VIE;
import static ru.yandex.direct.queryrec.model.Language.getLanguagesByAlphabetType;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.TextConstants.BEL_LETTERS;
import static ru.yandex.direct.utils.TextConstants.COMMON_UKR_BEL_LETTERS;
import static ru.yandex.direct.utils.TextConstants.GENERAL_TR_DE_LETTERS;
import static ru.yandex.direct.utils.TextConstants.KAZ_LETTERS;
import static ru.yandex.direct.utils.TextConstants.LAT_LETTERS;
import static ru.yandex.direct.utils.TextConstants.RUS_LETTERS;
import static ru.yandex.direct.utils.TextConstants.UKR_LETTERS;
import static ru.yandex.direct.utils.TextConstants.UNIQUE_DE_LETTERS;
import static ru.yandex.direct.utils.TextConstants.UNIQUE_TR_LETTERS;

/**
 * Сервис для распознавания языка на основе библиотеки queryrec
 */
@ParametersAreNonnullByDefault
public class QueryrecService {

    private static Logger logger = LoggerFactory.getLogger(QueryrecService.class);

    public static final Set<Long> ENABLE_UZBEK_LANGUAGE_FOR_ALL_CLIENTS_PROPERTY_VALUE = Set.of(-1L);

    /**
     * Возможные символы в английском, немецком, турецком языках, вхождение которых активирует распознавание языка
     * через queryec
     */
    private static final String COMMON_LATIN_LETTERS =
            LAT_LETTERS + GENERAL_TR_DE_LETTERS;

    private final LanguageRecognizer languageRecognizer;
    private final PpcProperty<Set<Long>> enableUzbekLanguageForClientsProperty;
    private final PpcProperty<Set<Long>> clientsWithdefaultVieLanguageProperty;
    private final PpcProperty<Set<String>> recognizedLanguagesProperty;
    private final UzbekLanguageThresholds uzbekLanguageThresholds;

    private final QueryrecJni queryrecJni;

    public QueryrecService(LanguageRecognizer languageRecognizer,
                           PpcProperty<Set<Long>> enableUzbekLanguageForClientsProperty,
                           PpcProperty<Set<Long>> enableVieLanguageForClientsProperty,
                           PpcProperty<Set<String>> recognizedLanguagesProperty,
                           UzbekLanguageThresholds uzbekLanguageThresholds,
                           QueryrecJni queryrecJni) {
        this.languageRecognizer = languageRecognizer;
        this.uzbekLanguageThresholds = uzbekLanguageThresholds;
        this.enableUzbekLanguageForClientsProperty = enableUzbekLanguageForClientsProperty;
        this.clientsWithdefaultVieLanguageProperty = enableVieLanguageForClientsProperty;
        this.recognizedLanguagesProperty = recognizedLanguagesProperty;
        this.queryrecJni = queryrecJni;
    }

    /**
     * Возвращает:
     * {@link Language} - язык из набора на основе вхождения национальных символов и страны клиента,
     * Language.UNKNOWN - если нет вхождения ни одного символа из допустимых алфавитов.
     *
     * @param text           исследуемый текст
     * @param clientId       id клиента
     * @param clientRegionId id региона клиента
     * @return Language - язык, результат распознавания
     */
    public Language recognize(@Nullable String text, @Nullable ClientId clientId, @Nullable Long clientRegionId) {
        if (StringUtils.isBlank(text)) {
            return UNKNOWN;
        }

        Set<Long> clientsWithEnabledVie = clientsWithdefaultVieLanguageProperty.getOrDefault(emptySet());
        if (clientId != null && clientsWithEnabledVie.contains(clientId.asLong())) {
            return VIE;
        }

        Set<Long> clientsWithEnabledUzbek = enableUzbekLanguageForClientsProperty.getOrDefault(emptySet());
        boolean isUzbekLanguageEnabled = (clientId != null && clientsWithEnabledUzbek.contains(clientId.asLong()))
                || clientsWithEnabledUzbek.equals(ENABLE_UZBEK_LANGUAGE_FOR_ALL_CLIENTS_PROPERTY_VALUE);

        if (!isUzbekLanguageEnabled) {
            return recognizeDeprecated(text);
        }

        // Если нашли кириллицу - скорее всего, это язык с кириллическим алфавитом. Смешение кириллицы и латиницы
        // чаще всего бывает, когда текст содержит непереведенные слова на латинице (например, "продам mercedes").
        if (ALL_CYRILLIC_LETTERS_MATCHER.matchesAnyOf(text)) {
            return recognizeCyrillicLanguage(text, clientRegionId);
        } else if (ALL_LATIN_LETTERS_MATCHER.matchesAnyOf(text)) {
            return recognizeLatinLanguage(text, clientRegionId);
        } else {
            return UNKNOWN;
        }
    }

    private Language recognizeCyrillicLanguage(String text, @Nullable Long clientRegionId) {
        Collection<Language> possibleLanguages = languageRecognizer.findPossibleCyrillicLanguagesByUniqueLetters(text);

        // Смешение 2 кириллических языков - крайне редкий случай, можно просто возвращать любой из этих языков.
        if (!possibleLanguages.isEmpty()) {
            if (possibleLanguages.size() > 1) {
                logger.warn("Recognized multiple languages for text: {}", text);
            }

            return possibleLanguages.iterator().next();
        }

        possibleLanguages = getLanguagesByAlphabetType(Alphabet.Type.CYRILLIC);
        Language fallbackLanguage = RUSSIAN;
        // Смешение 2 кириллических языков - редкость, поэтому если нашли символы, которых нет в русском алфавите,
        // то язык скорее всего не русский.
        if (ALL_CYRILLIC_LETTERS_EXCEPT_RUSSIAN_MATCHER.matchesAnyOf(text)) {
            Set<Character> nonRussianLetters = characterStream(text)
                    .filter(ALL_CYRILLIC_LETTERS_EXCEPT_RUSSIAN_MATCHER::matches)
                    .toImmutableSet();

            possibleLanguages = filterList(possibleLanguages,
                    language -> language.getNonBaseLettersByAlphabetType(Alphabet.Type.CYRILLIC)
                            .containsAll(nonRussianLetters));

            if (possibleLanguages.size() == 1) {
                return possibleLanguages.iterator().next();
            }

            fallbackLanguage = StreamEx.of(possibleLanguages)
                    .min(comparing(Language::getPriority))
                    .orElse(UNKNOWN);

            if (fallbackLanguage == UNKNOWN) {
                logger.warn("Can't recognize language for text: {}", text);
            }
        }

        Map<Language, Double> probabilityByLanguage = queryrecRecognize(text);
        Language bestAcceptableLanguage = getBestAcceptableLanguage(queryrecRecognize(text), possibleLanguages);

        if (isCyrillicUzbek(clientRegionId, possibleLanguages, probabilityByLanguage, bestAcceptableLanguage,
                uzbekLanguageThresholds)) {
            return UZBEK;
        }

        return nvl(bestAcceptableLanguage, fallbackLanguage);
    }

    private Language recognizeLatinLanguage(String text, @Nullable Long clientRegionId) {
        var recognizedLanguages = recognizedLanguagesProperty.getOrDefault(Set.of());
        var possibleLanguages = languageRecognizer.findPossibleLatinLanguagesByUniqueLetters(text,
                recognizedLanguages);

        if (possibleLanguages.size() == 1) {
            return possibleLanguages.iterator().next();
        }

        // Если нашли несколько языков по уникальным символам - скорее всего искомый язык среди них. Так бывает, когда
        // текст баннера содержит непереведенное слово на другом языке (например, название города на родном языке
        // этого города). Такое часто встречается с языками на латинице, гораздо реже с языками на кириллице.
        // Если не нашли ни одного - придется искать среди всех языков на латинице.
        if (possibleLanguages.isEmpty()) {
            possibleLanguages = getLanguagesByAlphabetType(Alphabet.Type.LATIN)
                    .stream()
                    .filter(language -> recognizedLanguages.isEmpty()
                            || recognizedLanguages.contains(language.getIso639Code().toString()))
                    .collect(Collectors.toSet());
        }

        Map<Language, Double> probabilityByLanguage = queryrecRecognize(text);
        Language bestAcceptableLanguage = getBestAcceptableLanguage(queryrecRecognize(text), possibleLanguages);

        if (isLatinUzbek(clientRegionId, possibleLanguages, probabilityByLanguage, bestAcceptableLanguage,
                uzbekLanguageThresholds)) {
            return UZBEK;
        }

        return nvl(bestAcceptableLanguage, ENGLISH);
    }

    /**
     * Устаревшая версия алгоритма распознавания языка.
     */
    @Deprecated
    public Language recognizeDeprecated(String text) {
        if (StringUtils.isBlank(text)) {
            return UNKNOWN;
        } else if (CharMatcher.anyOf(KAZ_LETTERS).matchesAnyOf(text)) {
            return KAZAKH;
        } else if (CharMatcher.anyOf(UKR_LETTERS).matchesAnyOf(text)) {
            return UKRAINIAN;
        } else if (CharMatcher.anyOf(BEL_LETTERS).matchesAnyOf(text)) {
            return BELARUSIAN;
        } else if (CharMatcher.anyOf(COMMON_UKR_BEL_LETTERS).matchesAnyOf(text)) {
            return getBestAcceptableLanguage(queryrecRecognize(text), asList(UKRAINIAN, BELARUSIAN), BELARUSIAN);
        } else if (CharMatcher.anyOf(RUS_LETTERS).matchesAnyOf(text)) {
            return RUSSIAN;
        } else if (CharMatcher.anyOf(UNIQUE_DE_LETTERS).matchesAnyOf(text)) {
            return GERMAN;
        } else if (CharMatcher.anyOf(UNIQUE_TR_LETTERS).matchesAnyOf(text)) {
            return TURKISH;
        } else if (CharMatcher.anyOf(COMMON_LATIN_LETTERS).matchesAnyOf(text)) {
            return getBestAcceptableLanguage(queryrecRecognize(text), asList(ENGLISH, TURKISH, GERMAN), ENGLISH);
        } else {
            return UNKNOWN;
        }
    }

    private Map<Language, Double> queryrecRecognize(String text) {
        try (TraceProfile profile = Trace.current().profile("queryrecRecognize")) {
            return queryrecJni.recognize(text);
        }
    }

    /**
     * @param text исследуемый текст
     * @return Соответствие языков и вероятностей, что текст написан на этом языке
     */
    public Map<Language, Double> getLangProbabilities(String text) {
        return queryrecRecognize(text);
    }
}
