package ru.yandex.direct.queryrec;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.queryrec.model.Language;

import static ru.yandex.direct.queryrec.model.Language.ENGLISH;
import static ru.yandex.direct.queryrec.model.Language.RUSSIAN;
import static ru.yandex.direct.queryrec.model.Language.TURKISH;
import static ru.yandex.direct.queryrec.model.Language.UNKNOWN;
import static ru.yandex.direct.queryrec.model.Language.UZBEK;
import static ru.yandex.direct.regions.Region.UZBEKISTAN_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
class QueryrecUtils {
    private QueryrecUtils() {
    }

    /**
     * Определяет по данным о тексте, является ли текст узбекским кириллическим.
     *
     * @param clientRegionId            id региона клиента
     * @param possibleLanguages         список возможных языков
     * @param probabilityByLanguage     мапа (язык -> вероятность, что текст принадлежит этому языку)
     * @param bestAcceptableLanguage    наиболее подходящий язык
     * @param thresholds                пороги для определения узбекского
     * @return true - если текст на узбекском, иначе false
     */
    static boolean isCyrillicUzbek(@Nullable Long clientRegionId,
                                   Collection<Language> possibleLanguages,
                                   Map<Language, Double> probabilityByLanguage,
                                   @Nullable Language bestAcceptableLanguage,
                                   UzbekLanguageThresholds thresholds) {
        if (!possibleLanguages.contains(UZBEK) || clientRegionId == null || clientRegionId != UZBEKISTAN_REGION_ID) {
            return false;
        }

        // При смешении русского и английского язык может быть ошибочно определен как английский.
        if (possibleLanguages.contains(RUSSIAN) && probabilityByLanguage.getOrDefault(ENGLISH, 0.0) == 1.0) {
            return false;
        }

        return (bestAcceptableLanguage == null || bestAcceptableLanguage == RUSSIAN)
                && probabilityByLanguage.getOrDefault(RUSSIAN, 0.0) < thresholds.getCyrillicUzbekToRussianThreshold();
    }

    /**
     * Определяет по данным о тексте, является ли текст узбекским латинским.
     *
     * @param clientRegionId            id региона клиента
     * @param possibleLanguages         список возможных языков
     * @param probabilityByLanguage     мапа (язык -> вероятность, что текст принадлежит этому языку)
     * @param bestAcceptableLanguage    наиболее подходящий язык
     * @param thresholds                пороги для определения узбекского
     * @return true - если текст на узбекском, иначе false
     */
    static boolean isLatinUzbek(@Nullable Long clientRegionId,
                                Collection<Language> possibleLanguages,
                                Map<Language, Double> probabilityByLanguage,
                                @Nullable Language bestAcceptableLanguage,
                                UzbekLanguageThresholds thresholds) {
        if (!possibleLanguages.contains(UZBEK) || clientRegionId == null || clientRegionId != UZBEKISTAN_REGION_ID) {
            return false;
        }

        // Скорее всего это один из языков, неподдержанных в Директе. В этом случае определяем язык как английский.
        if (bestAcceptableLanguage == ENGLISH
                && probabilityByLanguage.getOrDefault(UNKNOWN, 0.0) > thresholds.getUzbekToUnknownThreshold()) {
            return false;
        }

        return (bestAcceptableLanguage == ENGLISH
                        && probabilityByLanguage.getOrDefault(ENGLISH, 0.0) < thresholds.getLatinUzbekToRussianThreshold()) ||
                (bestAcceptableLanguage == TURKISH
                        && probabilityByLanguage.getOrDefault(TURKISH, 0.0) < thresholds.getUzbekToTurkishThreshold());
    }

    /**
     * Выбирает из множества вероятных языков наиболее веротный среди допустимых языков.
     *
     * @param langToProb      мапа с вероятностями языков
     * @param selectOnly      список языков, которые нас интересуют в мапе {@code langToProb}
     * @param defaultLanguage дефолтный язык, который нужно вернуть в случае, если в {@code langToProb} нет ни одного
     *                        языка из {@code selectOnly}
     * @return наиболее вероятный из допустимых языков, или {@code defaultLanguage}
     */
    static Language getBestAcceptableLanguage(Map<Language, Double> langToProb, List<Language> selectOnly,
                                              Language defaultLanguage) {
        return nvl(getBestAcceptableLanguage(langToProb, selectOnly), defaultLanguage);
    }

    @Nullable
    static Language getBestAcceptableLanguage(Map<Language, Double> langToProb, Collection<Language> selectOnly) {
        return langToProb.entrySet()
                .stream()
                .sorted(Map.Entry.<Language, Double>comparingByValue().reversed())
                .map(Map.Entry::getKey)
                .filter(selectOnly::contains)
                .findFirst()
                .orElse(null);
    }
}
