package ru.yandex.direct.queryrec.model;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import static com.google.common.collect.Lists.charactersOf;
import static java.util.Arrays.asList;
import static java.util.Arrays.stream;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.queryrec.QueryrecHelper.characterStream;

/**
 * Список доступных языков для распознавания, взят отсюда (остальные убраны, т.к. в директе не используются) :
 * https://a.yandex-team.ru/arc/trunk/arcadia/library/cpp/langs/langs.h
 */
@ParametersAreNonnullByDefault
public enum Language {

    UNKNOWN(0, "", ISO639Code.UNK, LanguagePriority.UNK),
    RUSSIAN(1, "ru", ISO639Code.RUS, LanguagePriority.RUS, Alphabet.RUSSIAN),
    ENGLISH(2, "en", ISO639Code.ENG, LanguagePriority.ENG, Alphabet.ENGLISH),
    UKRAINIAN(5, "uk", ISO639Code.UKR, LanguagePriority.UKR, Alphabet.UKRAINIAN),
    GERMAN(6, "de", ISO639Code.GER, LanguagePriority.GER, Alphabet.GERMAN),
    BELARUSIAN(9, "be", ISO639Code.BEL, LanguagePriority.BEL, Alphabet.BELARUSIAN),
    KAZAKH(10, "kk", ISO639Code.KAZ, LanguagePriority.KAZ, Alphabet.KAZAKH),
//    ESTONIAN(40, "et", ISO639Code.EST, Alphabet.ESTONIAN),
//    LATVIAN(41, "lv", ISO639Code.LAV, Alphabet.LATVIAN),
//    LITHUANIAN(42, "lt", ISO639Code.LIT, Alphabet.LITHUANIAN),
    TURKISH(44, "tr", ISO639Code.TUR, LanguagePriority.TUR, Alphabet.TURKISH),
    UZBEK(asList(47, 115), "uz", ISO639Code.UZB, LanguagePriority.UZB, Alphabet.UZBEK_CYRILLIC, Alphabet.UZBEK_LATIN),
    VIE(63, "vie", ISO639Code.VIE, LanguagePriority.VIE),
    SPANISH(12, "spa", ISO639Code.SPA, LanguagePriority.SPA, Alphabet.SPANISH),
    PORTUGUESE(16, "por", ISO639Code.POR, LanguagePriority.POR, Alphabet.PORTUGUESE),
    CZECH(24, "cze", ISO639Code.CZE, LanguagePriority.CZE, Alphabet.CZECH),
    POLISH(3, "pol", ISO639Code.POL, LanguagePriority.POL, Alphabet.POLISH),
    ;

    private static final Map<Integer, Language> VALUE_TO_LANG = StreamEx.of(Language.values())
            .mapToEntry(language -> language.values, identity())
            .flatMapKeys(StreamEx::of)
            .toImmutableMap();
    private static final Map<String, Language> NAME_TO_LANG = stream(Language.values())
            .collect(toMap(l -> l.name, identity()));
    private static final Map<Alphabet.Type, List<Language>> LANGUAGES_BY_ALPHABET_TYPE = StreamEx.of(Language.values())
            .mapToEntry(language -> language.alphabets, identity())
            .flatMapKeys(alphabets -> StreamEx.of(alphabets).map(Alphabet::getType))
            .grouping();

    private final List<Integer> values;
    private final String name;
    private final ISO639Code iso639Code;
    private final int priority;
    private final List<Alphabet> alphabets;

    private final Map<Alphabet.Type, ? extends Set<Character>> lettersByAlphabetType;
    private final Map<Alphabet.Type, Set<Character>> nonBaseLettersByAlphabetType;

    Language(int value, String name, ISO639Code iso639Code, LanguagePriority priority, Alphabet... alphabets) {
        this(singletonList(value), name, iso639Code, priority, alphabets);
    }

    Language(List<Integer> values, String name, ISO639Code iso639Code, LanguagePriority priority, Alphabet... alphabets) {
        this.values = values;
        this.name = name;
        this.iso639Code = iso639Code;
        this.priority = priority.ordinal();
        this.alphabets = asList(alphabets);

        lettersByAlphabetType = StreamEx.of(alphabets)
                .mapToEntry(Alphabet::getType,
                        alphabet -> ImmutableSet.copyOf(charactersOf(alphabet.getLetters())))
                .toImmutableMap();
        nonBaseLettersByAlphabetType = StreamEx.of(alphabets)
                .mapToEntry(Alphabet::getType,
                        alphabet -> characterStream(alphabet.getLetters())
                                .filter(letter -> !alphabet.getType().getBaseLetters().contains(letter))
                                .toImmutableSet())
                .toImmutableMap();
    }

    @Nonnull
    public static Language getByValue(int value) {
        return VALUE_TO_LANG.getOrDefault(value, UNKNOWN);
    }

    @Nonnull
    public static Language getByName(String name) {
        return NAME_TO_LANG.getOrDefault(name, UNKNOWN);
    }

    public List<Integer> getValues() {
        return values;
    }

    public String getName() {
        return name;
    }

    public ISO639Code getIso639Code() {
        return iso639Code;
    }

    public int getPriority() {
        return priority;
    }

    public boolean hasAlphabetOfType(Alphabet.Type type) {
        return lettersByAlphabetType.containsKey(type);
    }

    /**
     * @return все буквы из алфавита (заданного типа) языка в немутабельном виде.
     */
    public Set<Character> getAllLettersByAlphabetType(Alphabet.Type type) {
        return lettersByAlphabetType.get(type);
    }

    /**
     * @return все буквы, не содержащиеся в базовом алфавите заданного типа, в немутабельном виде.
     */
    public Set<Character> getNonBaseLettersByAlphabetType(Alphabet.Type type) {
        return nonBaseLettersByAlphabetType.get(type);
    }

    /**
     * Возвращает языки, которые имеют алфавит заданного типа {@link Alphabet.Type}
     *
     * @param alphabetType  тип алфавита
     * @return список языков
     */
    public static List<Language> getLanguagesByAlphabetType(Alphabet.Type alphabetType) {
        return LANGUAGES_BY_ALPHABET_TYPE.get(alphabetType);
    }
}
