package ru.yandex.direct.redislock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.redislock.clock.LockClock;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;

/**
 * Handy Lock which implements (spin)lock using {@link #tryLock()}
 */
public abstract class AbstractDistributedLock implements DistributedLock {
    private static final Logger logger = LoggerFactory.getLogger(AbstractDistributedLock.class);

    private final long lockAttemptTimeout;

    private boolean locked;

    private LockClock clock;

    protected AbstractDistributedLock(long lockAttemptTimeout) {
        this(lockAttemptTimeout, null);
    }

    protected AbstractDistributedLock(long lockAttemptTimeout, LockClock clock) {
        this.lockAttemptTimeout = lockAttemptTimeout;
        this.clock = clock == null ? LockClock.SYSTEM : clock;
    }


    @Override
    public boolean lock() throws DistributedLockException, InterruptedException {
        final long startTime = clock.nanoTime() / 1000_000;
        tryLock();
        long sleepTime;
        long remainingTime;
        int attemptNo = 0;
        while (!isLocked() && (remainingTime = calculateRemainingTime(lockAttemptTimeout, startTime)) > 0) {
            sleepTime = Math.min(ExponentialBackoffCalculator.INSTANCE.sleepTime(attemptNo), remainingTime);
            logger.debug("sleeping for {} msec", sleepTime);
            try (TraceChild traceProfile = Trace.current().child("redis-lock", "sleep")) {
                clock.sleep(sleepTime);
            }
            tryLock();
            ++attemptNo;
        }
        return isLocked();
    }

    private long calculateRemainingTime(long allowedMillis, long startTimeMillis) {
        final long executionTimeMillis = clock.nanoTime() / 1_000_000 - startTimeMillis;
        return allowedMillis - executionTimeMillis;
    }

    protected void setLocked(boolean locked) {
        this.locked = locked;
    }

    public boolean isLocked() {
        return locked;
    }
}
