package ru.yandex.direct.redislock;

import java.util.concurrent.ThreadLocalRandom;

/**
 * Encapsulates livelock exponential back-off timeout calculation.
 */
class ExponentialBackoffCalculator {
    public static final long DEFAULT_TIMEOUT_FIRST = 200L;
    public static final float DEFAULT_TIMEOUT_COEFF = 1.2f;
    public static final float DEFAULT_TIMEOUT_JITTER = 0.1f;

    public static final ExponentialBackoffCalculator INSTANCE = new ExponentialBackoffCalculator();


    private final long timeoutFirst;
    private final float timeoutCoeff;
    private final float timeoutJitter;

    public ExponentialBackoffCalculator() {
        this(DEFAULT_TIMEOUT_FIRST, DEFAULT_TIMEOUT_COEFF, DEFAULT_TIMEOUT_JITTER);
    }

    /**
     * @param timeoutFirst  first Timeout
     * @param timeoutCoeff  exponential back-off coeff
     * @param timeoutJitter exponential back-off jitter
     */
    public ExponentialBackoffCalculator(long timeoutFirst, float timeoutCoeff, float timeoutJitter) {
        this.timeoutFirst = timeoutFirst;
        this.timeoutCoeff = timeoutCoeff;
        this.timeoutJitter = timeoutJitter;
    }

    public long sleepTime(int attemptNo) {
        final double jitter =
                timeoutJitter > 0 ? (ThreadLocalRandom.current().nextDouble(timeoutJitter) - timeoutJitter / 2) : 0;
        return (long) (timeoutFirst * Math.pow(timeoutCoeff + jitter, attemptNo));
    }
}
