package ru.yandex.direct.redislock;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Holds various resources for JedisLock, in particular lua routines for redis.
 */
public class RedisLockRoutineLoader {

    public static final String LUA_ITERATIVE_LOCK_ROUTINE_FILE = "IterativeLock.lua";
    public static final String LUA_UNLOCK_ROUTINE_FILE = "Unlock.lua";

    private final Map<String, RedisScriptingRoutine> routineCache;

    public RedisLockRoutineLoader() {
        routineCache = new ConcurrentHashMap<>();
    }

    /**
     * Загружает скрипт Lua для блокировок
     *
     * @return {@link RedisScriptingRoutine} с соответствующим скриптом
     */
    public RedisScriptingRoutine getIterativeLockRoutine() {
        return getRoutine(LUA_ITERATIVE_LOCK_ROUTINE_FILE);
    }

    /**
     * Загружает скрипт Lua для разблокировок
     *
     * @return {@link RedisScriptingRoutine} с соответствующим скриптом
     */
    public RedisScriptingRoutine getUnlockRoutine() {
        return getRoutine(LUA_UNLOCK_ROUTINE_FILE);
    }

    /**
     * Загружает скрит Lua из файла
     *
     * @param name имя файла
     * @return {@link RedisScriptingRoutine} с соответствующим скриптом
     */
    public RedisScriptingRoutine getRoutine(String name) {
        if (!routineCache.containsKey(name)) {
            try {
                String routineBody = ResourceLoader.readResource("/" + name);
                routineCache.putIfAbsent(name, RedisScriptingRoutine.of(name, routineBody));
            } catch (IOException ex) {
                throw new RuntimeException("can't load " + name, ex);
            }
        }
        return routineCache.get(name);
    }
}
