package ru.yandex.direct.redislock;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Objects;

import javax.xml.bind.DatatypeConverter;

/**
 * Encapsulates redis (lua) routine as pair of
 * (sha1Digest digest; routine body plaintext).
 * Comes handy if redis answers NOSCRIPT when asked for 'evalsha sha1Digest';
 * in this case user should retry with 'eval script_body'.
 */
public class RedisScriptingRoutine {
    private final String sha1Digest;
    private final String routineBody;
    private final String name;

    private RedisScriptingRoutine(String sha1Digest, String name, String routineBody) {
        this.sha1Digest = sha1Digest;
        this.routineBody = routineBody;
        this.name = name;
    }

    public String getDigest() {
        return sha1Digest;
    }

    public String getRoutineBody() {
        return routineBody;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RedisScriptingRoutine routine = (RedisScriptingRoutine) o;
        return Objects.equals(sha1Digest, routine.sha1Digest) &&
                Objects.equals(routineBody, routine.routineBody) &&
                Objects.equals(name, routine.name);
    }

    @Override
    public int hashCode() {
        return Objects.hash(sha1Digest, routineBody, name);
    }

    @Override
    public String toString() {
        return "RedisScriptingRoutine{sha1Digest='" + sha1Digest + "\', name='" + name + "\'}";
    }

    public static RedisScriptingRoutine of(String name, String routineBody) {
        final String sha1Digest = computeSha1Digest(routineBody);
        return new RedisScriptingRoutine(sha1Digest, name, routineBody);
    }

    private static String computeSha1Digest(String from) {
        try {
            MessageDigest sha1Computer = MessageDigest.getInstance("SHA-1");
            byte[] scriptSha = sha1Computer.digest(from.getBytes(StandardCharsets.UTF_8));
            String ret = DatatypeConverter.printHexBinary(scriptSha);
            return ret;
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }
}
