package ru.yandex.direct.redislock.lettuce;

import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;
import java.util.stream.IntStream;

import io.lettuce.core.RedisException;
import io.lettuce.core.ScriptOutputType;
import io.lettuce.core.cluster.api.StatefulRedisClusterConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.redislock.AbstractDistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.redislock.RedisLockRoutineLoader;
import ru.yandex.direct.redislock.RedisScriptingRoutine;


/**
 * DistributedLock which uses redis. However, it acts as Semaphore if maxLocksNum > 1.
 * Each permit is paired with personal key 'keyPrefix{entry}permitNo' (and ttl). This complicated structure is required to
 * implement permit auto unlocking.
 * Actual key probing/unlocking is implemented with server-side lua script (RedisLockRoutineVerN.lua/RedisUnlockRoutine).
 */
public class LettuceLock extends AbstractDistributedLock {
    private static final Logger logger = LoggerFactory.getLogger(LettuceLock.class);

    private final Supplier<StatefulRedisClusterConnection<String, String>> connectionSupplier;
    private final RedisScriptingRoutine lockRoutine;
    private final RedisScriptingRoutine unlockRoutine;
    private final String[] keyBase;
    protected final int maxLocksNum;
    private final long ttl;
    private String lockedToKey;
    private String lockedByValue;

    public LettuceLock(Supplier<StatefulRedisClusterConnection<String, String>> connectionSupplier,
                       long lockAttemptTimeout,
                       long ttl,
                       String keyPrefix,
                       String entry,
                       int maxLocksNum,
                       RedisLockRoutineLoader routineLoader) {
        super(lockAttemptTimeout);
        this.connectionSupplier = connectionSupplier;
        this.lockRoutine = routineLoader.getIterativeLockRoutine();
        this.unlockRoutine = routineLoader.getUnlockRoutine();
        this.ttl = ttl;
        this.maxLocksNum = maxLocksNum;
        this.keyBase = new String[1];
        keyBase[0] = keyPrefix + "{" + entry + "}";
    }

    @Override
    public boolean tryLock() throws DistributedLockException {
        final String candidateValue = String.valueOf(ThreadLocalRandom.current().nextLong());
        try {
            Object lockResult = LettuceRoutineEvaluator.smartEval(connectionSupplier.get(), ScriptOutputType.STATUS,
                    lockRoutine, keyBase, candidateValue,
                    String.valueOf(ttl), String.valueOf(maxLocksNum));
            if (lockResult != null) {
                lockedToKey = lockResult.toString();
                lockedByValue = candidateValue;
            }
            setLocked(lockedToKey != null);
            return isLocked();
        } catch (RedisException ex) {
            setLocked(false);
            throw new DistributedLockException(ex);
        }
    }

    @Override
    public boolean unlock() throws DistributedLockException {
        try {
            if (!isLocked()) {
                logger.warn("lock is in non-locked state; unlock request was ignored");
            } else {
                String[] keys = {lockedToKey};
                LettuceRoutineEvaluator.smartEval(connectionSupplier.get(), ScriptOutputType.INTEGER, unlockRoutine,
                        keys, lockedByValue);
                setLocked(false);
                lockedToKey = null;
                lockedByValue = null;
            }
            return true;
        } catch (RedisException ex) {
            throw new DistributedLockException(ex);
        }
    }

    @Override
    public void unlockByEntry() {
        final String entryPrefix = keyBase[0];
        String[] keyValues =
                IntStream.range(1, maxLocksNum + 1).boxed().map(e -> entryPrefix + e).toArray(String[]::new);
        connectionSupplier.get().sync().del(keyValues);
    }
}
