package ru.yandex.direct.redislock.lettuce;

import java.util.function.Supplier;

import io.lettuce.core.cluster.api.StatefulRedisClusterConnection;

import ru.yandex.direct.redislock.DistributedLockBuilder;
import ru.yandex.direct.redislock.RedisLockRoutineLoader;

/**
 * Simplifies JedisLock instance creation
 */
public class LettuceLockBuilder implements DistributedLockBuilder {
    public static final int DEFAULT_MAX_LOCKS = 10;
    public static final long DEFAULT_LOCK_ATTEMPT_TIMEOUT = 3_000L;
    public static final long DEFAULT_TTL = 10_000L;
    public static final String DEFAULT_KEY_PREFIX = "lock";

    private final Supplier<StatefulRedisClusterConnection<String, String>> connectionSupplier;
    private final long lockAttemptTimeout;
    private final long ttl;
    private final int maxLocksNum;
    private final String keyPrefix;
    private final RedisLockRoutineLoader routineLoader;

    private LettuceLockBuilder(Supplier<StatefulRedisClusterConnection<String, String>> connectionSupplier) {
        this(connectionSupplier, DEFAULT_MAX_LOCKS, DEFAULT_LOCK_ATTEMPT_TIMEOUT, DEFAULT_TTL, DEFAULT_KEY_PREFIX,
                new RedisLockRoutineLoader());
    }


    private LettuceLockBuilder(Supplier<StatefulRedisClusterConnection<String, String>> connectionSupplier,
                               int maxLocksNum, long lockAttemptTimeout, long ttl,
                               String keyPrefix, RedisLockRoutineLoader routineLoader) {
        this.connectionSupplier = connectionSupplier;
        this.maxLocksNum = maxLocksNum;
        this.lockAttemptTimeout = lockAttemptTimeout;
        this.ttl = ttl;
        this.keyPrefix = keyPrefix;
        this.routineLoader = routineLoader;
    }

    public static LettuceLockBuilder newBuilder(
            Supplier<StatefulRedisClusterConnection<String, String>> connectionSupplier) {
        return new LettuceLockBuilder(connectionSupplier);
    }

    /**
     * @param newTimeout max timeout to acquire lock (milliseconds)
     * @return
     */
    public LettuceLockBuilder withLockAttemptTimeout(long newTimeout) {
        return new LettuceLockBuilder(connectionSupplier, maxLocksNum, newTimeout, ttl, keyPrefix, routineLoader);
    }

    /**
     * @param newTtl Semaphore's permit ttl (milliseconds)
     * @return
     */
    public LettuceLockBuilder withTTL(long newTtl) {
        return new LettuceLockBuilder(connectionSupplier, maxLocksNum, lockAttemptTimeout, newTtl, keyPrefix,
                routineLoader);
    }

    /**
     * @param newMaxLocks max allowed permits per lock key
     * @return
     */
    public LettuceLockBuilder withMaxLocks(int newMaxLocks) {
        return new LettuceLockBuilder(connectionSupplier, newMaxLocks, lockAttemptTimeout, ttl, keyPrefix,
                routineLoader);
    }

    /**
     * @param newPrefix keyPrefix to use
     * @return
     */
    public LettuceLockBuilder withKeyPrefix(String newPrefix) {
        return new LettuceLockBuilder(connectionSupplier, maxLocksNum, lockAttemptTimeout, ttl, newPrefix,
                routineLoader);
    }

    @Override
    public LettuceLock createLock(String lockKey) {
        return new LettuceLock(connectionSupplier,
                lockAttemptTimeout, ttl, keyPrefix, lockKey,
                maxLocksNum, routineLoader);
    }

    @Override
    public LettuceLock createLock(String lockKey, int maxLocks) {
        return new LettuceLock(connectionSupplier,
                lockAttemptTimeout, ttl, keyPrefix, lockKey,
                maxLocks, routineLoader);
    }

    public LettuceLock createLock(String lockKey, int maxLocks, long lockAttemptTimeout) {
        return new LettuceLock(connectionSupplier,
                lockAttemptTimeout, ttl, keyPrefix, lockKey,
                maxLocks, routineLoader);
    }
}
