package ru.yandex.direct.redislock.lettuce;

import io.lettuce.core.RedisCommandExecutionException;
import io.lettuce.core.RedisException;
import io.lettuce.core.ScriptOutputType;
import io.lettuce.core.cluster.api.StatefulRedisClusterConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.redislock.RedisScriptingRoutine;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

class LettuceRoutineEvaluator {
    private static final Logger logger = LoggerFactory.getLogger(LettuceRoutineEvaluator.class);

    private LettuceRoutineEvaluator() {
    }

    /**
     * Asks redis to evaluate lua script.
     * 1-st attempt: ask redis to eval script identified by sha1 digest(EVALSHA command).
     * 2-nd attempt: retry with EVAL+routine body if redis answers with 'NOSCRIPT' response.
     *
     * @param connectionInstance connectionInstance to send commands with
     * @param routine            script to execute
     * @param keys               script keys
     * @param args               script args
     */
    static Object smartEval(StatefulRedisClusterConnection<String, String> connectionInstance,
                            ScriptOutputType type,
                            RedisScriptingRoutine routine, String[] keys, String... args) {
        Object ret;
        try (TraceProfile profile = Trace.current().profile("redis:evalsha")) {
            ret = connectionInstance.sync().evalsha(routine.getDigest(), type, keys, args);
        } catch (RedisCommandExecutionException ex) {
            logger.info("caught NOSCRIPT from evalsha, about to attempt with eval");
            try (TraceProfile profile = Trace.current().profile("redis:eval")) {
                ret = connectionInstance.sync().eval(routine.getRoutineBody(), type, keys, args);
                logger.info("reattempt succeed");
            } catch (RedisException secondEx) {
                logger.info("reattempt failed", secondEx);
                throw ex;
            }
        } catch (RedisException ex) {
            logger.info("evalsha have found script, but failed during executing", ex);
            throw ex;
        }
        return ret;
    }
}
