package ru.yandex.direct.regions;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.regions.Region.CRIMEA_REGION_ID;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.regions.Region.REGION_TYPE_COUNTRY;
import static ru.yandex.direct.regions.Region.REGION_TYPE_PROVINCE;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@ParametersAreNonnullByDefault
public class GeoTree {
    private final Map<Long, Region> regions;
    private final Map<Long, Metro> metro;
    private final GeoTreeType geoTreeType;
    private final GeoTreeConstants geoTreeConstants;

    GeoTree(Map<Long, Region> regions, Map<Long, Metro> metro, GeoTreeType geoTreeType) {
        this.regions = regions;
        this.metro = metro;
        this.geoTreeType = geoTreeType;
        this.geoTreeConstants = new GeoTreeConstants(regions.values());
    }

    /**
     * Группирует минус-регионы из geoTargeting по их плюс-регионам
     *
     * @param geoTargeting - список плюс-регионов с их минус-регионами
     * @return map, где ключ - это id плюч-региона, а значение - это список id минус-регионов
     */
    static Map<Long, List<Long>> groupMinusRegionsByRegion(List<Long> geoTargeting) {
        if (geoTargeting.isEmpty()) {
            return emptyMap();
        }

        // предполагается что в geoTargeting минус-регионы всегда следуют за соответствующим
        // плюс-регионом, т.е. geoTargeting всегда должен начинаться с плюс-региона
        checkArgument(geoTargeting.get(0) >= 0,
                "incorrectly formed geoTargeting: " + geoTargeting.toString() + " - minus regions must follow region");

        // важно сохранить очередность - поэтому используется LinkedHashMap
        Map<Long, List<Long>> regionToMinusRegions = new LinkedHashMap<>();

        Long currentRegionId = null;
        for (Long regionId : geoTargeting) {
            if (regionId >= 0) { // region
                currentRegionId = regionId;
                regionToMinusRegions.put(currentRegionId, new ArrayList<>());
            } else { // minus region
                regionToMinusRegions.get(currentRegionId).add(regionId);
            }
        }

        return regionToMinusRegions;
    }

    /**
     * Проверка существования региона по id в геодереве
     */
    public boolean hasRegion(Long id) {
        return regions.containsKey(id);
    }

    /**
     * Получить регион по id
     */
    public Region getRegion(Long id) {
        return regions.get(id);
    }

    /**
     * Получить станцию метро по id
     */
    public Metro getMetro(Long id) {
        return metro.get(id);
    }

    /**
     * Проверка существования метро по id в геодереве
     */
    public boolean hasMetro(Long id) {
        return metro.containsKey(id);
    }

    /**
     * Получить map id - станция метро
     */
    public Map<Long, Metro> getMetroMap() {
        return Collections.unmodifiableMap(metro);
    }

    public GeoTreeType getGeoTreeType() {
        return geoTreeType;
    }

    /**
     * Проверяет вхождение одного региона в другой
     *
     * @param idToCheck         id региона, для которого проверяется принадлежность к региону-контейнеру
     * @param containerRegionId id региона-контейнера
     * @return true, если регион с id = idToCheck входит в регион containerRegionId
     */
    public boolean isRegionIncludedIn(long idToCheck, long containerRegionId) {
        return upRegionTo(idToCheck, singleton(containerRegionId)) == containerRegionId;
    }

    /*
     * Проверить что каждый регион из списка regions подчинен хотя бы одному региону из списка topRegions.

     * Минус регионы игнорируются
     */
    public boolean isRegionsIncludedIn(Collection<Long> regions, Set<Long> topRegions) {
        return regions.stream()
                .filter(id -> id >= 0L)
                .map(id -> upRegionTo(id, topRegions))
                .allMatch(topRegions::contains);
    }

    /*
     * Проверить что хотя бы один регион из списка regionIds подчинен хотя бы одному региону из списка topRegionIds
     *
     * См. GeoTools::is_targeting_in_region
     */
    public boolean isAnyRegionIncludedIn(Collection<Long> regionIds, Set<Long> topRegionIds) {
        if (regionIds.stream().noneMatch(id -> id >= 0)) {
            return true;
        }

        return regionIds.stream()
                .filter(id -> id >= 0)
                .map(id -> upRegionTo(id, topRegionIds))
                .anyMatch(topRegionIds::contains);
    }

    /**
     * Проверить что хотя бы один регион (или вложенный в него регион) из списка regionIds подчинен хотя бы одному
     * региону из списка topRegionIds (Т.е. содержится в одном из плюс-регионов и не содержится не в одном из
     * минус-регионов)
     * <p>
     * См. GeoTools::is_targeting_include_region
     */
    public boolean isAnyRegionOrSubRegionIncludedIn(Set<Long> regionIds, Set<Long> topRegionIds) {
        if (regionIds.isEmpty() || topRegionIds.isEmpty()) {
            return true;
        }

        Set<Long> plusTopRegionIds = topRegionIds.stream().filter(id -> id > 0).collect(toSet());
        Set<Long> minusTopRegionIds = topRegionIds.stream()
                .filter(id -> id < 0)
                .map(id -> -id)
                .collect(toSet());

        if (plusTopRegionIds.isEmpty()) {
            return true;
        }

        // Вначале проверям случай когда хотя бы один из regionIds вложен целиком в один из topRegionIds
        boolean isAnyRegionInsideAnyTopRegion = regionIds.stream()
                .map(Collections::singleton)
                .anyMatch(regionIdSet -> isAnyRegionIncludedIn(regionIdSet, plusTopRegionIds)
                        && !isAnyRegionIncludedIn(regionIdSet, minusTopRegionIds));
        if (isAnyRegionInsideAnyTopRegion) {
            return true;
        }

        // Проверяем случай когда хотя один из вложенных регионов из списка regionIds содержится в topRegionIds
        return isAnyRegionIncludedIn(plusTopRegionIds, regionIds);
    }


    /**
     * Получение списка тех регионов regionIds, которые будут в таргетинге по переданным topRegionIds
     * При этом игнорируется случай, когда регион из regionIds совпадает с плюс-регионом из topRegionIds
     */
    public Set<Long> getIncludedRegions(Set<Long> regionIds, List<Long> topRegionIds) {
        if (regionIds.isEmpty() || topRegionIds.isEmpty()) {
            return emptySet();
        }

        Set<Long> plusTopRegionIds = topRegionIds.stream().filter(id -> id >= 0).collect(toSet());
        Set<Long> minusTopRegionIds = topRegionIds.stream()
                .filter(id -> id < 0)
                .map(id -> -id)
                .collect(toSet());
        return regionIds.stream()
                .filter(region -> {
                    if (!isRegionsIncludedIn(singletonList(region), plusTopRegionIds) ||
                            plusTopRegionIds.contains(region)) {
                        return false;
                    }
                    long minusRegion = upRegionTo(region, minusTopRegionIds);
                    long plusRegion = upRegionTo(region, plusTopRegionIds);

                    return minusRegion == GLOBAL_REGION_ID || !isRegionIncludedIn(minusRegion, plusRegion);
                })
                .collect(Collectors.toSet());
    }

    /**
     * Получить прародителя региона из переданных или GLOBAL_REGION_ID если прародитель не входит
     *
     * @param regionToUp регион, для которого ищем прародителя верхнего уровня
     * @param type       тип региона верхнего уровня, до которых пробуем подняться вверх по дереву
     * @return прародитель региона со значением типа, равным type, иначе GLOBAL_REGION_ID
     */
    public long upRegionToType(long regionToUp, Integer type) {
        Region currentRegion = regions.get(regionToUp);
        while (currentRegion != null && currentRegion.getId() != GLOBAL_REGION_ID) {
            if (type.equals(currentRegion.getType())) {
                return currentRegion.getId();
            }

            currentRegion = currentRegion.getParent();
        }

        return GLOBAL_REGION_ID;
    }

    /**
     * Получить прародителя региона из переданных или GLOBAL_REGION_ID если прародитель не входит в переданные
     *
     * @param regionToUp регион, для которого ищем прародителя верхнего уровня
     * @param topRegions регионы верхнего уровня, до которых пробуем подняться вверх по дереву
     * @return прародитель региона: один из указанных в topRegions, если совпал, иначе GLOBAL_REGION_ID
     */
    public long upRegionTo(long regionToUp, Set<Long> topRegions) {
        Region currentRegion = regions.get(regionToUp);
        while (currentRegion != null && currentRegion.getId() != GLOBAL_REGION_ID) {
            if (topRegions.contains(currentRegion.getId())) {
                return currentRegion.getId();
            }

            currentRegion = currentRegion.getParent();
        }

        return GLOBAL_REGION_ID;
    }

    /**
     * Вычисляет геофлаг баннера по гео группы
     * Если у всех гео выставлен флаг, то выставляет флаг для баннера
     * (См. GeoTools::refine_geoid)
     *
     * @param geo список гео группы
     * @return результат сохранения геофлага для баннера
     */
    public boolean hasGeoFlagByGeo(@Nullable Collection<Long> geo) {
        if (geo == null || geo.isEmpty()) {
            return false;
        }

        // Если переданы только минус регионы, то используем геофлаг 0-ого региона
        if (geo.stream().noneMatch(id -> id > 0)) {
            return regions.get(0L).getGeoFlag();
        }

        return geo.stream()
                // Исключаем минус-регионы, так как они в любом случае не влияют на geoflag
                .filter(id -> id > 0)
                .allMatch(id -> {
                    Region region = regions.get(id);
                    return region != null && region.getGeoFlag();
                });
    }

    /**
     * Проверяет существует ли станция метро с данным идентификатором в заданном городе
     * <p>
     * См. protected/Metro.pm::metro_list и Validation/VCards.pm::validate_metro
     */
    public boolean isCityHasMetro(String city, Long metroGeoId) {
        Objects.requireNonNull(city, "city");
        Objects.requireNonNull(metroGeoId, "metroGeoId");

        Metro metroStation = metro.get(metroGeoId);
        if (metroStation == null) {
            return false;
        }

        Region metroStationCity = regions.get(metroStation.getCityGeoId());

        checkState(metroStationCity != null, "Not found city for metro %d", metroGeoId);

        return metroStationCity.equalsAnyNameIgnoreCase(city);
    }

    // TODO : temp method
    public Map<Long, Region> getRegions() {
        return regions;
    }

    /**
     * Вычитает из указанного списка регионов регионы, указанные в втором списке
     *
     * @param geoTargeting     список регионов, упорядоченный по уменьшению регионов, может содержать минус-регионы
     * @param regionsToExclude список гео регионов, которые нужно исключить
     * @return получившийся список регионов
     */
    public List<Long> excludeRegions(List<Long> geoTargeting, List<Long> regionsToExclude) {

        // TODO: сделать поддержку параметра return_nonempty_geo из perl-версии

        if (regionsToExclude.isEmpty()) {
            return geoTargeting;
        }

        List<Long> geo = new ArrayList<>(geoTargeting);
        List<Long> exclude = new ArrayList<>(regionsToExclude);

        if (geo.isEmpty()) {
            geo.add(GLOBAL_REGION_ID);
        }

        Map<Long, List<Long>> regionToMinusRegions = groupMinusRegionsByRegion(geo);

        Deque<Long> resultGeo = new ArrayDeque<>();
        Set<Long> usedMinusRegion = new HashSet<>();

        // обходим получившийся список регионов в обратном порядке - от малых регионов в к большим
        ListIterator<Map.Entry<Long, List<Long>>> mapIterator =
                new ArrayList<>(regionToMinusRegions.entrySet()).listIterator(regionToMinusRegions.size());

        while (mapIterator.hasPrevious()) {
            Map.Entry<Long, List<Long>> entry = mapIterator.previous();

            Long region = entry.getKey();
            List<Long> minusRegions = entry.getValue();

            List<Long> resultMinusRegion = new ArrayList<>(minusRegions);

            if (!minusRegions.isEmpty()) {
                // если регион, который хотят исключить, является подрегионом минус-региона, то убираем из исключений
                // этот регион, например:
                //      (Россия, -Центр) минус (Москва, Спб) -- Москву убираем из исключений, т.к. уже стоит -Центр
                Set<Long> idFromMinusRegions = minusRegions.stream().map(Math::abs).collect(toSet());
                exclude = exclude.stream().filter(e -> !isRegionsIncludedIn(singleton(e), idFromMinusRegions))
                        .collect(toList());
            }

            if (!exclude.isEmpty()) {
                // если регион хотят явно исключить, то не включаем такой регион в результат, например:
                //      (Россия, Москва) минус (Москва) -- выкидываем Москву
                if (isRegionsIncludedIn(singleton(region), new HashSet<>(exclude))) {
                    continue;
                }

                // если регион, который хотят исключить, является подрегионом региона, то добавляем в минус-регионы,
                // например:
                //      (Россия, Центр) минус (МосОбласть) -- (Россия, Центр, -МосОбласть)
                List<Long> effectiveExcludes =
                        exclude.stream().filter(e -> isRegionIncludedIn(e, region)).map(e -> -e).collect(toList());

                resultMinusRegion.addAll(effectiveExcludes);
            }

            resultMinusRegion.removeAll(usedMinusRegion);
            // TODO: нужно ли добавить поддержку уникальности?

            usedMinusRegion.addAll(resultMinusRegion);

            // соединяем снова в обратном порядке - крупные регионы с относящимимся к ним минус-регионами в начале
            ListIterator<Long> listIterator = resultMinusRegion.listIterator(resultMinusRegion.size());
            while (listIterator.hasPrevious()) {
                resultGeo.addFirst(listIterator.previous());
            }
            resultGeo.addFirst(region);
        }

        return new ArrayList<>(resultGeo);
    }

    /**
     * Добавляет в указанный список регионов регионы, указанные в втором списке
     *
     * @param geoTargeting     список регионов, упорядоченный по уменьшению регионов, может содержать минус-регионы
     * @param regionsToInclude список гео регионов, которые нужно добавить, должен содержать только плюс-регионы
     * @return получившийся список регионов
     */
    public List<Long> includeRegions(List<Long> geoTargeting, List<Long> regionsToInclude) {

        if (regionsToInclude.isEmpty()) {
            return geoTargeting;
        }

        List<Long> include = regionsToInclude;

        Map<Long, List<Long>> regionToMinusRegions = groupMinusRegionsByRegion(geoTargeting);

        Deque<Long> resultGeo = new ArrayDeque<>();

        // обходим получившийся список регионов в обратном порядке - от малых регионов в к большим
        ListIterator<Map.Entry<Long, List<Long>>> mapIterator =
                new ArrayList<>(regionToMinusRegions.entrySet()).listIterator(regionToMinusRegions.size());

        Set<Long> addedRegions = new HashSet<>();

        while (mapIterator.hasPrevious()) {
            Map.Entry<Long, List<Long>> entry = mapIterator.previous();

            Long region = entry.getKey();
            List<Long> minusRegions = entry.getValue();

            List<Long> resultMinusRegion = new ArrayList<>(minusRegions);

            if (!include.isEmpty()) {
                // 1) если текущий регион является подрегионом одного из добавляемых - пропускаем его,
                // он будет добавлен уровнем выше
                if (isRegionsIncludedIn(singleton(region), new HashSet<>(include))) {
                    continue;
                }

                // добавляемые регионы, которые содержатся в текущем
                List<Long> regionsToAddInCurrent = filterList(include, e -> isRegionIncludedIn(e, region));
                addedRegions.addAll(regionsToAddInCurrent);

                // 2) убираем минус регионы у текущего, которые содержатся в добавляемых регионах
                // т.к. эти минус регионы включаются в добавляемые, они больше не исключаются из дерева
                resultMinusRegion = filterList(minusRegions, e -> !isRegionsIncludedIn(singleton(-e),
                        new HashSet<>(regionsToAddInCurrent)));

                Set<Long> resultMinusRegionPositiveIds = listToSet(resultMinusRegion, e -> -e);

                // 3) добавляем регионы. причем нужно добавить только те, которые исключаются минус регионами текущего.
                // Добавляем как новый блок из одного +региона
                // А те регионы, которые минус регионами не исключаются, добавлять не требуется, они
                // и так содержатся в исходном дереве
                List<Long> effectiveRegionsToAdd = filterList(regionsToAddInCurrent,
                        e -> isRegionsIncludedIn(singleton(e), resultMinusRegionPositiveIds));

                ListIterator<Long> listIterator =
                        effectiveRegionsToAdd.listIterator(effectiveRegionsToAdd.size());
                while (listIterator.hasPrevious()) {
                    resultGeo.addFirst(listIterator.previous());
                }

                // удаляем из списка добавляемых уже добавленные регионы
                include = filterList(include, e -> !addedRegions.contains(e));
            }

            // соединяем снова в обратном порядке - крупные регионы с относящимимся к ним минус-регионами в начале
            ListIterator<Long> listIterator = resultMinusRegion.listIterator(resultMinusRegion.size());
            while (listIterator.hasPrevious()) {
                resultGeo.addFirst(listIterator.previous());
            }
            resultGeo.addFirst(region);
        }

        if (!include.isEmpty()) {
            resultGeo.addAll(include);
        }

        return new ArrayList<>(resultGeo);
    }

    /**
     * Находит те регионы, которые есть в первом списке и которых нет во втором списке (NB: без учета вложенности
     * регионов!). Регионы в результирующем списке расширяются/сужаются до уровня страны, например:
     * <p>
     * get_geo_diff("Минск,Москва", "Москва") = 'Беларусь'
     * get_geo_diff("Весь_Мир", "Весь_Мир,-Беларусь") = 'Беларусь'
     *
     * @param firstGeo  первый список гео регионов
     * @param secondGeo второй список гео регионов
     * @return получившийся список регионов
     */
    public List<Long> getDiffScaledToCountryLevel(List<Long> firstGeo, List<Long> secondGeo) {
        Predicate<Long> isNegative = e -> e < 0L;
        Predicate<Long> notNegative = e -> e >= 0L;

        // уникальные минус-регионы из первого списка регионов
        Set<Long> firstGeoMinusRegions = firstGeo.stream().filter(isNegative).collect(toSet());

        // уникальные плюс-регионы из второго списка регионов
        Set<Long> secondGeoPlusRegions = secondGeo.stream().filter(notNegative).collect(toSet());

        // плюc-регионы, которые есть только в первом списке
        List<Long> resultRegions =
                firstGeo.stream().filter(e -> notNegative.test(e) && !secondGeoPlusRegions.contains(e))
                        .collect(toList());

        // minus-регионы, которые есть только во втором списке
        List<Long> resultMinusRegions =
                secondGeo.stream().filter(e -> isNegative.test(e) && !firstGeoMinusRegions.contains(e))
                        .collect(toList());

        resultRegions.addAll(resultMinusRegions.stream().map(Math::abs).collect(toList()));

        return resultRegions.stream().map(region -> {
            long countryRegion = upRegionToType(region, REGION_TYPE_COUNTRY);
            if (countryRegion == GLOBAL_REGION_ID &&
                    upRegionToType(region, REGION_TYPE_PROVINCE) == CRIMEA_REGION_ID) {
                // если страна региона не определена и при этом провинция региона - Крым, то расширяем до Крыма.
                // актуально например для API-шного геодерева.
                return CRIMEA_REGION_ID;
            }
            return countryRegion;
        }).filter(e -> e != GLOBAL_REGION_ID).distinct().collect(toList());
    }

    /**
     * Причесать ID регионов, чтобы после каждого плюс-региона шли минус-регионы, которые в него входят
     * <p>
     * В результате плюс-регионы упорядочены по возрастанию, минус-регионы после каждого плюс-региона
     * упорядочены по возрастанию их модуля
     *
     * @param geoIds список регионов: плюс-регионы обозначены неотрицательными числами, минус-регионы отрицательными
     *               метод не пытается что-то менять в этом списке
     * @return новый список, в котором регионы идут в правильном порядке
     * @throws IllegalStateException если какой-то минус-регион не входит ни в один плюс-регион
     */
    public List<Long> refineGeoIds(List<Long> geoIds) {
        Set<Long> plusRegions = geoIds.stream()
                .filter(id -> id >= 0)
                .collect(toCollection(LinkedHashSet::new));

        boolean globalRegionInPlusRegions = plusRegions.contains(GLOBAL_REGION_ID);

        Map<Long, Set<Long>> plusRegionToMinusRegions = geoIds.stream()
                .filter(id -> id < 0)
                .collect(groupingBy(
                        minusRegion -> {
                            Long plusRegion = upRegionTo(-minusRegion, plusRegions);

                            if (!globalRegionInPlusRegions && plusRegion.equals(GLOBAL_REGION_ID)) {
                                throw new IllegalArgumentException(
                                        "no plus region found for minus region of " + minusRegion);
                            }

                            return plusRegion;
                        },
                        toCollection(LinkedHashSet::new)));

        return plusRegions.stream()
                .flatMap(plusRegion -> plusRegionToMinusRegions.containsKey(plusRegion) ?
                        Stream.concat(Stream.of(plusRegion), plusRegionToMinusRegions.get(plusRegion).stream())
                        : Stream.of(plusRegion))
                .collect(toList());
    }

    /**
     * Формирует список стран по списку гео
     * <ul>
     * <li>Для объектов выше страны - перечисляет все входящие туда страны</li>
     * <li>Для объектов ниже страны - добавляет в список ту страну, куда они входят</li>
     * <li>Для объектов ниже страны, которые при этом не входят ни в одну страну (например, Крым в дереве API),
     * перечисляет все известные страны.</li>
     * </ul>
     */
    public Set<Long> getModerationCountries(Collection<Long> geoIds) {
        if (geoIds.isEmpty()) {
            return emptySet();
        }

        // Если среди списка регионов попался
        if (geoIds.contains(GLOBAL_REGION_ID)) {
            return geoTreeConstants.getCountriesIds();
        }

        Set<Long> result = new HashSet<>();
        for (Long geoId : geoIds) {
            if (geoId <= 0) {
                // Учитываются только плюс-регионы
                continue;
            }
            long scaledUpId = upRegionToType(geoId, REGION_TYPE_COUNTRY);
            if (scaledUpId != GLOBAL_REGION_ID) {
                result.add(scaledUpId);
            } else {
                Set<Long> countriesToAdd = new HashSet<>();
                for (Long country : geoTreeConstants.getCountriesIds()) {
                    if (upRegionTo(country, singleton(geoId)) != GLOBAL_REGION_ID) {
                        countriesToAdd.add(country);
                    }
                }
                if (!countriesToAdd.isEmpty()) {
                    result.addAll(countriesToAdd);
                } else {
                    // Если регион не пренадлежит ни одной стране - добавляем все известные страны и выходим
                    result.addAll(geoTreeConstants.getCountriesIds());
                    break;
                }
            }
        }

        return result;
    }

    public GeoTreeConstants getGeoTreeConstants() {
        return geoTreeConstants;
    }

    public Set<Long> getAllChildren(long regionId) {
        var set = new HashSet<Long>();
        var children = getChildren(regionId);
        set.addAll(children);
        for (Long childId : children) {
            set.addAll(getAllChildren(childId));
        }
        return set;
    }

    public Set<Long> getChildren(long regionId) {
        return regions
                .values()
                .stream()
                .filter(region -> region.getParent().getId() == regionId).map(Region::getId)
                .collect(Collectors.toSet());
    }

    /**
     * Возвращает id региона вместе с id всех его родителей
     */
    public Set<Long> regionIdWithParents(Long id) {
        if (id == null || id == 0L) {//Специальный регион Все. У него в родителях он сам
            return emptySet();
        }
        var region = getRegion(id);
        var set = new HashSet<Long>();
        while (region.getId() > 0) {
            set.add(region.getId());
            region = region.getParent();
        }
        return set;
    }

    public boolean isOneRegionsIncludeAnotherRegions(List<Long> regions, List<Long> regionsThatCouldBeInclude) {
        Set<Long> regionsSet = new HashSet<>(regions);
        Set<Long> regionsThatCouldBeIncludeSet = new HashSet<>(regionsThatCouldBeInclude);
        for (Long regionId: regionsThatCouldBeInclude) {
            if (regionId < 0) {
                continue;
            }
            Region region = getRegion(regionId);
            while (region != null) {
                if (regionsSet.contains(-region.getId())) {
                    return false;
                }
                if (regionsSet.contains(region.getId())) {
                    break;
                } else {
                    region = region.getParent();
                }
            }
            if (region == null) {
                return false;
            }
        }

        for (Long regionId: regions) {
            if (regionId > 0) {
                continue;
            }
            Region region = getRegion(-regionId);
            while (region != null) {
                if (regionsThatCouldBeIncludeSet.contains(region.getId())) {
                    return false;
                }
                if (regionsThatCouldBeIncludeSet.contains(-region.getId())) {
                    break;
                } else {
                    region = region.getParent();
                }
            }
            if (region == null) {
                return false;
            }
        }
        return true;
    }
}
